"""Unit tests for MercurialClient."""

from __future__ import annotations

import os
import re
import shutil
import tempfile
import time
import unittest
from random import randint
from textwrap import dedent

import kgb
from rbtools.diffs.errors import ApplyPatchError

import rbtools.helpers
from rbtools.api.resource import FileAttachmentItemResource
from rbtools.clients import RepositoryInfo
from rbtools.clients.errors import (CreateCommitError,
                                    MergeError,
                                    SCMClientDependencyError,
                                    SCMError)
from rbtools.clients.mercurial import MercurialClient, MercurialRefType
from rbtools.clients.tests import (FOO, FOO1, FOO2, FOO3, FOO4, FOO5, FOO6,
                                   SCMClientTestCase)
from rbtools.config.loader import load_config
from rbtools.diffs.patches import BinaryFilePatch, Patch, PatchAuthor
from rbtools.testing.api.transport import URLMapTransport
from rbtools.utils.checks import check_install
from rbtools.utils.filesystem import (is_exe_in_path,
                                      make_tempdir)
from rbtools.utils.process import (RunProcessResult,
                                   run_process,
                                   run_process_exec)


hgext_path = os.path.abspath(os.path.join(rbtools.helpers.__file__, '..',
                                          'hgext.py'))


class MercurialTestCase(SCMClientTestCase[MercurialClient]):
    """Base class for all Mercurial unit tests."""

    TESTSERVER = 'http://127.0.0.1:8080'

    CLONE_HGRC = dedent("""
        [ui]
        username = test user <user at example.com>

        [paths]
        default = %(hg_dir)s
        cloned = %(clone_dir)s

        [reviewboard]
        url = %(test_server)s

        [diff]
        git = true
    """).rstrip()

    scmclient_cls = MercurialClient
    populate_default_clone: bool = True
    default_scmclient_caps = {
        'scmtools': {
            'mercurial': {
                'empty_files': True,
            },
        },
    }

    #: The path to the source test repository directory.
    hg_dir: str

    ######################
    # Instance variables #
    ######################

    #: The path to the Mercurial clone directory for a test.
    clone_dir: str | None

    @classmethod
    def setup_checkout(
        cls,
        checkout_dir: str,
    ) -> str | None:
        """Populate a Mercurial clone.

        This will create a clone of the sample Mercurial repository stored in
        the :file:`testdata` directory.

        Args:
            checkout_dir (str):
                The top-level directory in which the clone will be placed.

        Returns:
            str:
            The main clone directory, or ``None`` if :command:`hg` isn't
            in the path.
        """
        if cls.populate_default_clone:
            client = MercurialClient()

            if not client.has_dependencies():
                return None

            cls.hg_dir = os.path.join(cls.testdata_dir, 'hg-repo')
            cls.run_hg(['clone', '--stream', cls.hg_dir, checkout_dir])

            return checkout_dir
        else:
            return super().setup_checkout(checkout_dir)

    def setUp(self) -> None:
        """Set up state for a unit test.

        This will write a Mercurial configuration file suitable for unit tests
        to the clone directory, if set.
        """
        super().setUp()

        self.clone_dir = self.checkout_dir

        if self.clone_dir is not None:
            self.clone_hgrc_path = os.path.join(self.clone_dir, '.hg', 'hgrc')

            with open(self.clone_hgrc_path, 'w', encoding='utf-8') as fp:
                fp.write(self.CLONE_HGRC % {
                    'hg_dir': self.hg_dir,
                    'clone_dir': self.clone_dir,
                    'test_server': self.TESTSERVER,
                })
        else:
            self.clone_dir = None

    @classmethod
    def run_hg(
        cls,
        command: list[str],
        **kwargs,
    ) -> RunProcessResult:
        """Run a Mercurial command.

        Args:
            command (list of str):
                The command and arguments to pass to :program:`hg`.

            **kwargs (dict):
                Additional keyword arguments to pass to
                :py:func:`~rbtools.utils.process.run_process`.

        Returns:
            rbtools.utils.process.RunProcessResult:
            The result of :py:func:`~rbtools.utils.process.run_process`.
        """
        return run_process(
            ['hg', *command],
            env={
                'HGPLAIN': '1',
            },
            **kwargs)

    def hg_add_file_commit(
        self,
        filename: str = 'test.txt',
        data: bytes = b'Test',
        msg: str = 'Test commit',
        branch: (str | None) = None,
        bookmark: (str | None) = None,
        tag: (str | None) = None,
        date: (str | None) = None,
    ) -> None:
        """Add a file to the repository and commit it.

        This can also optionally construct a branch for the commit.

        Args:
            filename (str, optional):
                The name of the file to write.

            data (bytes, optional):
                The data to write to the file.

            msg (str, optional):
                The commit message.

            branch (str, optional):
                The optional branch to create.

            bookmark (str, optional):
                The optional bookmark to create.

            tag (str, optional):
                The optional tag to create.

            date (str, optional):
                The optional date to set for the commit.

                Version Added:
                    4.0
        """
        if branch:
            self.run_hg(['branch', branch])

        if bookmark:
            self.run_hg(['bookmark', bookmark])

        with open(filename, 'wb') as f:
            f.write(data)

        commit_args = ['commit', '-A', '-m', msg]

        if date:
            commit_args += ['-d', date]

        commit_args.append(filename)

        self.run_hg(commit_args)

        if tag:
            self.run_hg(['tag', tag])

    def hg_get_num_commits(self) -> int:
        """Return the number of commits in the repository.

        Returns:
            int:
            The number of commits.
        """
        return len(
            self.run_hg(['log', '--template', 'X\\n'])
            .stdout
            .readlines()
        )

    def _hg_get_tip(self) -> str:
        """Return the revision at the tip of the branch.

        Returns:
            str:
            The tip revision.
        """
        # This is currently the most cross-Mercurial way of getting a
        # full ID.
        cmdline = ['identify', '--debug']

        return (
            self.run_hg(cmdline)
            .stdout
            .read()
            .split()[0]
        )


class MercurialClientTests(MercurialTestCase):
    """Unit tests for MercurialClient."""

    AUTHOR = PatchAuthor(full_name='name',
                         email='email')

    def test_check_dependencies_with_found(self):
        """Testing MercurialClient.check_dependencies with hg found"""
        self.spy_on(check_install, op=kgb.SpyOpMatchAny([
            {
                'args': (['hg', '--help'],),
                'op': kgb.SpyOpReturn(True),
            },
        ]))

        client = self.build_client(setup=False)
        client.check_dependencies()

        self.assertSpyCallCount(check_install, 1)
        self.assertSpyCalledWith(check_install, ['hg', '--help'])

    def test_check_dependencies_with_missing(self):
        """Testing MercurialClient.check_dependencies with dependencies
        missing
        """
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        message = "Command line tools ('hg') are missing."

        with self.assertRaisesMessage(SCMClientDependencyError, message):
            client.check_dependencies()

        self.assertSpyCallCount(check_install, 1)
        self.assertSpyCalledWith(check_install, ['hg', '--help'])

    def test_get_local_path_with_deps_missing(self) -> None:
        """Testing MercurialClient.get_local_path with dependencies missing"""
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        # Make sure dependencies are checked for this test before we run
        # get_local_path(). This will be the expected setup flow.
        self.assertFalse(client.has_dependencies())

        with self.assertLogs(level='DEBUG') as ctx:
            local_path = client.get_local_path()

        self.assertIsNone(local_path)

        self.assertEqual(ctx.records[0].msg,
                         'Unable to execute "hg --help": skipping Mercurial')

        self.assertSpyCallCount(check_install, 1)
        self.assertSpyCalledWith(check_install, ['hg', '--help'])

    def test_get_local_path_with_deps_not_checked(self) -> None:
        """Testing MercurialClient.get_local_path with dependencies not
        checked
        """
        # A False value is used just to ensure get_local_path() bails early,
        # and to minimize side-effects.
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        message = re.escape(
            'Either MercurialClient.setup() or '
            'MercurialClient.has_dependencies() must be called before other '
            'functions are used.'
        )

        with self.assertRaisesRegex(SCMError, message):
            client.get_local_path()

    def test_get_repository_info(self):
        """Testing MercurialClient.get_repository_info"""
        client = self.build_client()
        ri = client.get_repository_info()

        self.assertIsInstance(ri, RepositoryInfo)
        self.assertEqual(ri.base_path, '')

        hgpath = ri.path

        if os.path.basename(hgpath) == '.hg':
            hgpath = os.path.dirname(hgpath)

        self.assertEqual(self.hg_dir, hgpath)

    def test_get_repository_info_with_deps_missing(self) -> None:
        """Testing MercurialClient.get_repository_info with dependencies
        missing
        """
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        # Make sure dependencies are checked for this test before we run
        # get_repository_info(). This will be the expected setup flow.
        self.assertFalse(client.has_dependencies())

        with self.assertLogs(level='DEBUG') as ctx:
            repository_info = client.get_repository_info()

        self.assertIsNone(repository_info)

        self.assertEqual(ctx.records[0].msg,
                         'Unable to execute "hg --help": skipping Mercurial')

        self.assertSpyCallCount(check_install, 1)
        self.assertSpyCalledWith(check_install, ['hg', '--help'])

    def test_get_repository_info_with_deps_not_checked(self) -> None:
        """Testing MercurialClient.get_repository_info with dependencies
        not checked
        """
        # A False value is used just to ensure get_repository_info() bails
        # early, and to minimize side-effects.
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        message = re.escape(
            'Either MercurialClient.setup() or '
            'MercurialClient.has_dependencies() must be called before other '
            'functions are used.'
        )

        with self.assertRaisesRegex(SCMError, message):
            client.get_repository_info()

    def test_scan_for_server(self):
        """Testing MercurialClient.scan_for_server"""
        client = self.build_client()
        client.hgrc = {}

        assert self.clone_dir is not None
        os.rename(self.clone_hgrc_path,
                  os.path.join(self.clone_dir, '._disabled_hgrc'))

        client._load_hgrc()

        ri = client.get_repository_info()

        self.assertIsNone(client.scan_for_server(ri))

    def test_scan_for_server_when_present_in_hgrc(self):
        """Testing MercurialClient.scan_for_server when present in hgrc"""
        client = self.build_client()
        ri = client.get_repository_info()

        self.assertEqual(client.scan_for_server(ri),
                         self.TESTSERVER)

    def test_scan_for_server_reviewboardrc(self):
        """Testing MercurialClient.scan_for_server when in .reviewboardrc"""
        client = self.build_client()

        with self.reviewboardrc({'REVIEWBOARD_URL': self.TESTSERVER}):
            client.config = load_config()
            ri = client.get_repository_info()

            self.assertEqual(client.scan_for_server(ri),
                             self.TESTSERVER)

    def test_diff(self):
        """Testing MercurialClient.diff"""
        client = self.build_client(needs_diff=True)

        base_commit_id = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='delete and modify stuff')
        commit_id = self._hg_get_tip()

        revisions = client.parse_revision_spec([])

        spy = self.spy_on(client._execute)
        result = client.diff(revisions)

        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-r', base_commit_id, '-r', commit_id],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)

        self.assertIsInstance(result, dict)
        self.assertEqual(result, {
            'base_commit_id': base_commit_id,
            'commit_id': commit_id,
            'diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -6,7 +6,4 @@\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            ) % {
                b'base_commit_id': base_commit_id.encode('utf-8'),
                b'commit_id': commit_id.encode('utf-8'),
            },
            'parent_diff': None,
        })

    def test_diff_with_multiple_commits(self):
        """Testing MercurialClient.diff with multiple commits"""
        client = self.build_client(needs_diff=True)

        base_commit_id = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO3,
                                msg='commit 3')
        commit_id = self._hg_get_tip()

        revisions = client.parse_revision_spec([])

        spy = self.spy_on(client._execute)
        result = client.diff(revisions)

        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-r', base_commit_id, '-r', commit_id],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)

        self.assertEqual(result, {
            'base_commit_id': base_commit_id,
            'commit_id': commit_id,
            'diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -1,12 +1,11 @@\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b' Italiam, fato profugus, Laviniaque venit\n'
                b' litora, multum ille et terris iactatus et alto\n'
                b' vi superum saevae memorem Iunonis ob iram;\n'
                b'-multa quoque et bello passus, dum conderet urbem,\n'
                b'+dum conderet urbem,\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b'+Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            ) % {
                b'base_commit_id': base_commit_id.encode('utf-8'),
                b'commit_id': commit_id.encode('utf-8'),
            },
            'parent_diff': None,
        })

    def test_diff_with_exclude_patterns(self):
        """Testing MercurialClient.diff with exclude_patterns"""
        client = self.build_client(needs_diff=True)

        base_commit_id = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')
        self.hg_add_file_commit(filename='exclude.txt',
                                data=FOO2,
                                msg='commit 2')
        commit_id = self._hg_get_tip()

        revisions = client.parse_revision_spec([])

        spy = self.spy_on(client._execute)
        result = client.diff(revisions,
                             exclude_patterns=['exclude.txt'])

        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-X', 'exclude.txt', '-r', base_commit_id, '-r', commit_id],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)

        self.assertEqual(result, {
            'base_commit_id': base_commit_id,
            'commit_id': commit_id,
            'diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -6,7 +6,4 @@\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            ) % {
                b'base_commit_id': base_commit_id.encode('utf-8'),
                b'commit_id': commit_id.encode('utf-8'),
            },
            'parent_diff': None,
        })

    def test_diff_with_exclude_patterns_with_empty_file(self):
        """Testing MercurialClient.diff with exclude_patterns matching empty
        file
        """
        client = self.build_client(needs_diff=True)

        base_commit_id = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')
        self.hg_add_file_commit(filename='empty.txt',
                                data=b'',
                                msg='commit 2')
        commit_id = self._hg_get_tip()

        revisions = client.parse_revision_spec([])

        spy = self.spy_on(client._execute)
        result = client.diff(revisions,
                             exclude_patterns=['empty.txt'])

        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-X', 'empty.txt', '-r', base_commit_id, '-r', commit_id],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)

        self.assertEqual(result, {
            'base_commit_id': base_commit_id,
            'commit_id': commit_id,
            'diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -6,7 +6,4 @@\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            ) % {
                b'base_commit_id': base_commit_id.encode('utf-8'),
                b'commit_id': commit_id.encode('utf-8'),
            },
            'parent_diff': None,
        })

    def test_diff_with_diverged_branch(self):
        """Testing MercurialClient.diff with diverged branch"""
        client = self.build_client(needs_diff=True)

        base_commit_id = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')
        commit_id1 = self._hg_get_tip()

        # Create a "diverged" branch and make a commit there. We'll start by
        # generating a diff for this commit.
        self.run_hg(['branch', 'diverged'])
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2')
        commit_id2 = self._hg_get_tip()

        revisions = client.parse_revision_spec([])

        spy = self.spy_on(client._execute)
        result = client.diff(revisions)

        self.assertEqual(len(spy.calls), 2)
        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-r', base_commit_id, '-r', commit_id1],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)
        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-r', commit_id1, '-r', commit_id2],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)

        self.assertEqual(result, {
            'base_commit_id': base_commit_id,
            'commit_id': commit_id2,
            'diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -1,3 +1,5 @@\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b' Italiam, fato profugus, Laviniaque venit\n'
                b' litora, multum ille et terris iactatus et alto\n'
            ) % {
                b'base_commit_id': commit_id1.encode('utf-8'),
                b'commit_id': commit_id2.encode('utf-8'),
            },
            'parent_diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -6,7 +6,4 @@\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            ) % {
                b'base_commit_id': base_commit_id.encode('utf-8'),
                b'commit_id': commit_id1.encode('utf-8'),
            },
        })

        # Switch back to the default branch.
        self.run_hg(['update', '-C', 'default'])
        self.assertEqual(commit_id1, self._hg_get_tip())

        revisions = client.parse_revision_spec([])

        spy.reset_calls()
        result = client.diff(revisions)

        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-r', base_commit_id, '-r', commit_id1],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)

        self.assertEqual(len(spy.calls), 1)
        self.assertEqual(result, {
            'base_commit_id': base_commit_id,
            'commit_id': commit_id1,
            'diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -6,7 +6,4 @@\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            ) % {
                b'base_commit_id': base_commit_id.encode('utf-8'),
                b'commit_id': commit_id1.encode('utf-8'),
            },
            'parent_diff': None,
        })

    def test_diff_with_parent_diff(self):
        """Testing MercurialClient.diff with parent diffs"""
        client = self.build_client(needs_diff=True)

        base_commit_id = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2')
        commit_id2 = self._hg_get_tip()

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO3,
                                msg='commit 3')
        commit_id3 = self._hg_get_tip()

        revisions = client.parse_revision_spec(['2', '3'])

        spy = self.spy_on(client._execute)
        result = client.diff(revisions)

        self.assertEqual(len(spy.calls), 2)
        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-r', commit_id2, '-r', commit_id3],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)
        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-r', base_commit_id, '-r', commit_id2],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)

        self.assertEqual(result, {
            'base_commit_id': base_commit_id,
            'commit_id': None,
            'diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -1,11 +1,11 @@\n'
                b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b' Italiam, fato profugus, Laviniaque venit\n'
                b' litora, multum ille et terris iactatus et alto\n'
                b' vi superum saevae memorem Iunonis ob iram;\n'
                b'-multa quoque et bello passus, dum conderet urbem,\n'
                b'+dum conderet urbem,\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b'+Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b' \n'
            ) % {
                b'base_commit_id': commit_id2.encode('utf-8'),
                b'commit_id': commit_id3.encode('utf-8'),
            },
            'parent_diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -1,3 +1,5 @@\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b' Italiam, fato profugus, Laviniaque venit\n'
                b' litora, multum ille et terris iactatus et alto\n'
                b'@@ -6,7 +8,4 @@\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            ) % {
                b'base_commit_id': base_commit_id.encode('utf-8'),
                b'commit_id': commit_id2.encode('utf-8'),
            },
        })

    def test_diff_with_parent_diff_and_diverged_branch(self):
        """Testing MercurialClient.diff with parent diffs and diverged branch
        """
        # This test is very similar to test_diff_with_parent_diff except we
        # throw a branch into the mix.
        client = self.build_client(needs_diff=True)

        base_commit_id = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')

        self.run_hg(['branch', 'diverged'])
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2')
        commit_id2 = self._hg_get_tip()

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO3,
                                msg='commit 3')
        commit_id3 = self._hg_get_tip()

        revisions = client.parse_revision_spec(['2', '3'])

        spy = self.spy_on(client._execute)
        result = client.diff(revisions)

        self.assertEqual(len(spy.calls), 2)
        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-r', commit_id2, '-r', commit_id3],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)
        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-r', base_commit_id, '-r', commit_id2],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)

        self.assertEqual(result, {
            'base_commit_id': base_commit_id,
            'commit_id': None,
            'diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -1,11 +1,11 @@\n'
                b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b' Italiam, fato profugus, Laviniaque venit\n'
                b' litora, multum ille et terris iactatus et alto\n'
                b' vi superum saevae memorem Iunonis ob iram;\n'
                b'-multa quoque et bello passus, dum conderet urbem,\n'
                b'+dum conderet urbem,\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b'+Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b' \n'
            ) % {
                b'base_commit_id': commit_id2.encode('utf-8'),
                b'commit_id': commit_id3.encode('utf-8'),
            },
            'parent_diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -1,3 +1,5 @@\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b' Italiam, fato profugus, Laviniaque venit\n'
                b' litora, multum ille et terris iactatus et alto\n'
                b'@@ -6,7 +8,4 @@\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            ) % {
                b'base_commit_id': base_commit_id.encode('utf-8'),
                b'commit_id': commit_id2.encode('utf-8'),
            },
        })

    def test_diff_with_parent_diff_using_option(self):
        """Testing MercurialClient.diff with parent diffs using --parent"""
        # This test is very similar to test_diff_with_parent_diff except we
        # use the --parent option to post without explicit revisions
        client = self.build_client(
            needs_diff=True,
            options={
                'parent_branch': '2',
            })

        base_commit_id = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2')
        commit_id2 = self._hg_get_tip()

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO3,
                                msg='commit 3')
        commit_id3 = self._hg_get_tip()

        revisions = client.parse_revision_spec([])

        spy = self.spy_on(client._execute)
        result = client.diff(revisions)

        self.assertEqual(len(spy.calls), 2)
        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-r', commit_id2, '-r', commit_id3],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)
        self.assertSpyCalledWith(
            spy,
            ['hg', 'diff', '--hidden', '--nodates', '-g',
             '-r', base_commit_id, '-r', commit_id2],
            env={
                'HGPLAIN': '1',
            },
            log_debug_output_on_error=False)

        self.assertEqual(result, {
            'base_commit_id': base_commit_id,
            'commit_id': commit_id3,
            'diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -1,11 +1,11 @@\n'
                b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b' Italiam, fato profugus, Laviniaque venit\n'
                b' litora, multum ille et terris iactatus et alto\n'
                b' vi superum saevae memorem Iunonis ob iram;\n'
                b'-multa quoque et bello passus, dum conderet urbem,\n'
                b'+dum conderet urbem,\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b'+Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b' \n'
            ) % {
                b'base_commit_id': commit_id2.encode('utf-8'),
                b'commit_id': commit_id3.encode('utf-8'),
            },
            'parent_diff': (
                b'# HG changeset patch\n'
                b'# Node ID %(commit_id)s\n'
                b'# Parent  %(base_commit_id)s\n'
                b'diff --git a/foo.txt b/foo.txt\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -1,3 +1,5 @@\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b' Italiam, fato profugus, Laviniaque venit\n'
                b' litora, multum ille et terris iactatus et alto\n'
                b'@@ -6,7 +8,4 @@\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            ) % {
                b'base_commit_id': base_commit_id.encode('utf-8'),
                b'commit_id': commit_id2.encode('utf-8'),
            },
        })

    def test_parse_revision_spec_with_no_args(self):
        """Testing MercurialClient.parse_revision_spec with no arguments"""
        client = self.build_client()

        base = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2')

        tip = self._hg_get_tip()

        self.assertEqual(
            client.parse_revision_spec([]),
            {
                'base': base,
                'commit_id': tip,
                'tip': tip,
            })

    def test_parse_revision_spec_with_one_arg_periods(self):
        """Testing MercurialClient.parse_revision_spec with r1..r2 syntax"""
        client = self.build_client()

        base = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')

        tip = self._hg_get_tip()

        self.assertEqual(
            client.parse_revision_spec(['0..1']),
            {
                'base': base,
                'tip': tip,
            })

    def test_parse_revision_spec_with_one_arg_colons(self):
        """Testing MercurialClient.parse_revision_spec with r1::r2 syntax"""
        client = self.build_client()

        base = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')

        tip = self._hg_get_tip()

        self.assertEqual(
            client.parse_revision_spec(['0..1']),
            {
                'base': base,
                'tip': tip,
            })

    def test_parse_revision_spec_with_one_arg(self):
        """Testing MercurialClient.parse_revision_spec with one revision"""
        client = self.build_client()

        base = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')
        tip = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2')

        self.assertEqual(
            client.parse_revision_spec(['1']),
            {
                'base': base,
                'commit_id': tip,
                'tip': tip,
            })

    def test_parse_revision_spec_with_two_args(self):
        """Testing MercurialClient.parse_revision_spec with two revisions"""
        client = self.build_client()

        base = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2')
        tip = self._hg_get_tip()

        self.assertEqual(
            client.parse_revision_spec(['0', '2']),
            {
                'base': base,
                'tip': tip,
            })

    def test_parse_revision_spec_with_parent_base(self):
        """Testing MercurialClient.parse_revision_spec with parent base"""
        client = self.build_client()

        start_base = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')
        commit1 = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2')
        commit2 = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO3,
                                msg='commit 3')
        commit3 = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO4,
                                msg='commit 4')
        commit4 = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO5,
                                msg='commit 5')

        self.assertEqual(
            client.parse_revision_spec(['1', '2']),
            {
                'base': commit1,
                'tip': commit2,
                'parent_base': start_base,
            })

        self.assertEqual(
            client.parse_revision_spec(['4']),
            {
                'base': commit3,
                'tip': commit4,
                'parent_base': start_base,
                'commit_id': commit4,
            })

        self.assertEqual(
            client.parse_revision_spec(['2', '4']),
            {
                'base': commit2,
                'tip': commit4,
                'parent_base': start_base,
            })

    def test_get_hg_ref_type(self):
        """Testing MercurialClient.get_hg_ref_type"""
        client = self.build_client()

        self.hg_add_file_commit(branch='test-branch',
                                bookmark='test-bookmark',
                                tag='test-tag')
        tip = self._hg_get_tip()

        self.assertEqual(client.get_hg_ref_type('test-branch'),
                         MercurialRefType.BRANCH)
        self.assertEqual(client.get_hg_ref_type('test-bookmark'),
                         MercurialRefType.BOOKMARK)
        self.assertEqual(client.get_hg_ref_type('test-tag'),
                         MercurialRefType.TAG)
        self.assertEqual(client.get_hg_ref_type(tip),
                         MercurialRefType.REVISION)
        self.assertEqual(client.get_hg_ref_type('something-invalid'),
                         MercurialRefType.UNKNOWN)

    def test_get_commit_message_with_one_commit_in_range(self):
        """Testing MercurialClient.get_commit_message with range containing
        only one commit
        """
        client = self.build_client(options={
            'guess_description': True,
            'guess_summary': True,
        })

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1')

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.get_commit_message(revisions),
            {
                'description': 'commit 1',
                'summary': 'commit 1',
            })

    def test_get_commit_message_with_commit_range(self):
        """Testing MercurialClient.get_commit_message with commit range"""
        client = self.build_client(options={
            'guess_description': True,
            'guess_summary': True,
        })

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1\n\ndesc1')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2\n\ndesc2')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO3,
                                msg='commit 3\n\ndesc3')

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.get_commit_message(revisions),
            {
                'description': (
                    'desc1\n'
                    '\n'
                    'commit 2\n'
                    '\n'
                    'desc2\n'
                    '\n'
                    'commit 3\n'
                    '\n'
                    'desc3'
                ),
                'summary': 'commit 1',
            })

    def test_get_commit_message_with_specific_commit(self):
        """Testing MercurialClient.get_commit_message with specific commit"""
        client = self.build_client(options={
            'guess_description': True,
            'guess_summary': True,
        })

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1\n\ndesc1')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2\n\ndesc2')
        tip = self._hg_get_tip()
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO3,
                                msg='commit 3\n\ndesc3')

        revisions = client.parse_revision_spec([tip])

        self.assertEqual(
            client.get_commit_message(revisions),
            {
                'description': 'desc2',
                'summary': 'commit 2',
            })

    def test_commit_history(self):
        """Testing MercurialClient.get_commit_history"""
        client = self.build_client()

        base_commit_id = self._hg_get_tip()

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='commit 1\n\ndesc1',
                                date='2022-08-04T11:00:00-07:00')
        commit_id1 = self._hg_get_tip()

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO2,
                                msg='commit 2\n\ndesc2',
                                date='2022-08-05T12:00:00-07:00')
        commit_id2 = self._hg_get_tip()

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO3,
                                msg='commit 3\n\ndesc3',
                                date='2022-08-06T13:00:00-07:00')
        commit_id3 = self._hg_get_tip()

        revisions = client.parse_revision_spec([])
        commit_history = client.get_commit_history(revisions)

        self.assertEqual(
            commit_history,
            [
                {
                    'author_date': '2022-08-04T11:00:00-07:00',
                    'author_email': 'user at example.com',
                    'author_name': 'test user <user at example.com>',
                    'commit_id': commit_id1,
                    'commit_message': 'commit 1\n\ndesc1',
                    'parent_id': base_commit_id,
                },
                {
                    'author_date': '2022-08-05T12:00:00-07:00',
                    'author_email': 'user at example.com',
                    'author_name': 'test user <user at example.com>',
                    'commit_id': commit_id2,
                    'commit_message': 'commit 2\n\ndesc2',
                    'parent_id': commit_id1,
                },
                {
                    'author_date': '2022-08-06T13:00:00-07:00',
                    'author_email': 'user at example.com',
                    'author_name': 'test user <user at example.com>',
                    'commit_id': commit_id3,
                    'commit_message': 'commit 3\n\ndesc3',
                    'parent_id': commit_id2,
                },
            ])

    def test_create_commit_with_run_editor_true(self):
        """Testing MercurialClient.create_commit with run_editor set to True"""
        client = self.build_client()
        self.spy_on(client._execute)

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        client.create_commit(message='Test commit message.',
                             author=self.AUTHOR,
                             run_editor=True,
                             files=['foo.txt'])

        self.assertSpyLastCalledWith(
            client._execute,
            ['hg', 'commit', '-m', 'TEST COMMIT MESSAGE.', '-u',
             'name <email>', 'foo.txt'])

    def test_create_commit_with_run_editor_false(self):
        """Testing MercurialClient.create_commit with run_editor set to False
        """
        client = self.build_client()
        self.spy_on(client._execute)

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        client.create_commit(message='Test commit message.',
                             author=self.AUTHOR,
                             run_editor=False,
                             files=['foo.txt'])

        self.assertSpyLastCalledWith(
            client._execute,
            ['hg', 'commit', '-m', 'Test commit message.', '-u',
             'name <email>', 'foo.txt'])

    def test_create_commit_with_all_files_true(self):
        """Testing MercurialClient.create_commit with all_files set to True"""
        client = self.build_client()
        self.spy_on(client._execute)

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        client.create_commit(message='message',
                             author=self.AUTHOR,
                             run_editor=False,
                             files=[],
                             all_files=True)

        self.assertSpyLastCalledWith(
            client._execute,
            ['hg', 'commit', '-m', 'message', '-u', 'name <email>', '-A'])

    def test_create_commit_with_all_files_false(self):
        """Testing MercurialClient.create_commit with all_files set to False"""
        client = self.build_client()
        self.spy_on(client._execute)

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        client.create_commit(message='message',
                             author=self.AUTHOR,
                             run_editor=False,
                             files=['foo.txt'],
                             all_files=False)

        self.assertSpyLastCalledWith(
            client._execute,
            ['hg', 'commit', '-m', 'message', '-u', 'name <email>',
             'foo.txt'])

    def test_create_commit_with_empty_commit_message(self):
        """Testing MercurialClient.create_commit with empty commit message"""
        client = self.build_client()

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        message = (
            "A commit message wasn't provided. The patched files are in "
            "your tree but haven't been committed."
        )

        with self.assertRaisesMessage(CreateCommitError, message):
            client.create_commit(message='',
                                 author=self.AUTHOR,
                                 run_editor=True,
                                 files=['foo.txt'])

    def test_create_commit_without_author(self):
        """Testing MercurialClient.create_commit without author information"""
        client = self.build_client()
        self.spy_on(client._execute)

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        client.create_commit(message='Test commit message.',
                             author=None,
                             run_editor=True,
                             files=['foo.txt'])

        self.assertSpyLastCalledWith(
            client._execute,
            ['hg', 'commit', '-m', 'TEST COMMIT MESSAGE.', 'foo.txt'])

    def test_merge_with_branch_and_close_branch_false(self):
        """Testing MercurialClient.merge with target branch and
        close_branch=False
        """
        client = self.build_client()

        self.hg_add_file_commit(branch='test-branch')

        self.spy_on(client._execute)
        client.merge(target='test-branch',
                     destination='default',
                     message='My merge commit',
                     author=self.AUTHOR,
                     close_branch=False)

        calls = client._execute.calls
        self.assertSpyCallCount(client._execute, 5)
        self.assertSpyCalledWith(calls[0], ['hg', 'log', '-ql1', '-r',
                                            'bookmark(test-branch)'])
        self.assertSpyCalledWith(calls[1], ['hg', 'branches', '-q'])
        self.assertSpyCalledWith(calls[2], ['hg', 'update', 'default'])
        self.assertSpyCalledWith(calls[3], ['hg', 'merge', 'test-branch'])
        self.assertSpyCalledWith(calls[4], ['hg', 'commit', '-m',
                                            'My merge commit',
                                            '-u', 'name <email>'])

    def test_merge_with_branch_and_close_branch_true(self):
        """Testing MercurialClient.merge with target branch and
        close_branch=True
        """
        client = self.build_client()

        self.hg_add_file_commit(branch='test-branch')

        self.spy_on(client._execute)
        client.merge(target='test-branch',
                     destination='default',
                     message='My merge commit',
                     author=self.AUTHOR,
                     close_branch=True)

        calls = client._execute.calls
        self.assertSpyCallCount(client._execute, 7)
        self.assertSpyCalledWith(calls[0], ['hg', 'log', '-ql1', '-r',
                                            'bookmark(test-branch)'])
        self.assertSpyCalledWith(calls[1], ['hg', 'branches', '-q'])
        self.assertSpyCalledWith(calls[2], ['hg', 'update', 'test-branch'])
        self.assertSpyCalledWith(calls[3], ['hg', 'commit', '-m',
                                            'My merge commit',
                                            '--close-branch'])
        self.assertSpyCalledWith(calls[4], ['hg', 'update', 'default'])
        self.assertSpyCalledWith(calls[5], ['hg', 'merge', 'test-branch'])
        self.assertSpyCalledWith(calls[6], ['hg', 'commit', '-m',
                                            'My merge commit',
                                            '-u', 'name <email>'])

    def test_merge_with_bookmark_and_close_branch_false(self):
        """Testing MercurialClient.merge with target bookmark and
        close_branch=False
        """
        client = self.build_client()

        self.run_hg(['branch', 'feature-work'])
        self.hg_add_file_commit(bookmark='test-bookmark')

        self.spy_on(client._execute)
        client.merge(target='test-bookmark',
                     destination='default',
                     message='My merge commit',
                     author=self.AUTHOR,
                     close_branch=False)

        calls = client._execute.calls
        self.assertSpyCallCount(client._execute, 4)
        self.assertSpyCalledWith(calls[0], ['hg', 'log', '-ql1', '-r',
                                            'bookmark(test-bookmark)'])
        self.assertSpyCalledWith(calls[1], ['hg', 'update', 'default'])
        self.assertSpyCalledWith(calls[2], ['hg', 'merge', 'test-bookmark'])
        self.assertSpyCalledWith(calls[3], ['hg', 'commit', '-m',
                                            'My merge commit',
                                            '-u', 'name <email>'])

    def test_merge_with_bookmark_and_close_branch_true(self):
        """Testing MercurialClient.merge with target bookmark and
        close_branch=True
        """
        client = self.build_client()

        self.run_hg(['branch', 'feature-work'])
        self.hg_add_file_commit(bookmark='test-bookmark')

        self.spy_on(client._execute)
        client.merge(target='test-bookmark',
                     destination='default',
                     message='My merge commit',
                     author=self.AUTHOR,
                     close_branch=True)

        calls = client._execute.calls
        self.assertSpyCallCount(client._execute, 5)
        self.assertSpyCalledWith(calls[0], ['hg', 'log', '-ql1', '-r',
                                            'bookmark(test-bookmark)'])
        self.assertSpyCalledWith(calls[1], ['hg', 'update', 'default'])
        self.assertSpyCalledWith(calls[2], ['hg', 'merge', 'test-bookmark'])
        self.assertSpyCalledWith(calls[3], ['hg', 'commit', '-m',
                                            'My merge commit',
                                            '-u', 'name <email>'])
        self.assertSpyCalledWith(calls[4], ['hg', 'bookmark', '-d',
                                            'test-bookmark'])

    def test_merge_with_tag(self):
        """Testing MercurialClient.merge with target tag"""
        client = self.build_client()

        self.run_hg(['branch', 'feature-work'])
        self.hg_add_file_commit(tag='test-tag')

        self.spy_on(client._execute)
        client.merge(target='test-tag',
                     destination='default',
                     message='My merge commit',
                     author=self.AUTHOR,
                     close_branch=True)

        calls = client._execute.calls
        self.assertSpyCallCount(client._execute, 6)
        self.assertSpyCalledWith(calls[0], ['hg', 'log', '-ql1', '-r',
                                            'bookmark(test-tag)'])
        self.assertSpyCalledWith(calls[1], ['hg', 'branches', '-q'])
        self.assertSpyCalledWith(calls[2], ['hg', 'log', '-ql1', '-r',
                                            'tag(test-tag)'])
        self.assertSpyCalledWith(calls[3], ['hg', 'update', 'default'])
        self.assertSpyCalledWith(calls[4], ['hg', 'merge', 'test-tag'])
        self.assertSpyCalledWith(calls[5], ['hg', 'commit', '-m',
                                            'My merge commit',
                                            '-u', 'name <email>'])

    def test_merge_with_revision(self):
        """Testing MercurialClient.merge with target revision"""
        client = self.build_client()

        self.run_hg(['branch', 'feature-work'])
        self.hg_add_file_commit()
        tip = self._hg_get_tip()

        self.spy_on(client._execute)
        client.merge(target=tip,
                     destination='default',
                     message='My merge commit',
                     author=self.AUTHOR,
                     close_branch=True)

        calls = client._execute.calls
        self.assertSpyCallCount(client._execute, 7)
        self.assertSpyCalledWith(calls[0], ['hg', 'log', '-ql1', '-r',
                                            'bookmark(%s)' % tip])
        self.assertSpyCalledWith(calls[1], ['hg', 'branches', '-q'])
        self.assertSpyCalledWith(calls[2], ['hg', 'log', '-ql1', '-r',
                                            'tag(%s)' % tip])
        self.assertSpyCalledWith(calls[3], ['hg', 'identify', '-r', tip])
        self.assertSpyCalledWith(calls[4], ['hg', 'update', 'default'])
        self.assertSpyCalledWith(calls[5], ['hg', 'merge', tip])
        self.assertSpyCalledWith(calls[6], ['hg', 'commit', '-m',
                                            'My merge commit',
                                            '-u', 'name <email>'])

    def test_merge_with_invalid_target(self):
        """Testing MercurialClient.merge with an invalid target"""
        client = self.build_client()

        expected_message = (
            'Could not find a valid branch, tag, bookmark, or revision called '
            '"invalid".'
        )

        with self.assertRaisesMessage(MergeError, expected_message):
            client.merge(target='invalid',
                         destination='default',
                         message='commit message',
                         author=self.AUTHOR)

    def test_merge_with_invalid_destination(self):
        """Testing MercurialClient.merge with an invalid destination branch"""
        client = self.build_client()

        expected_message = 'Could not switch to branch "non-existent-branch".'

        with self.assertRaisesMessage(MergeError, expected_message):
            client.merge(target='default',
                         destination='non-existent-branch',
                         message='commit message',
                         author=self.AUTHOR)

    def test_apply_patch(self):
        """Testing MercurialClient.apply_patch"""
        client = self.build_client()

        self.spy_on(run_process_exec,
                    op=kgb.SpyOpReturn((0, b'test', b'')))

        result = client.apply_patch(patch_file='test.diff',
                                    p=None,
                                    base_path='',
                                    base_dir='')

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', 'test.diff', '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])

        self.assertTrue(result.applied)
        self.assertFalse(result.has_conflicts)
        self.assertEqual(result.conflicting_files, [])
        self.assertEqual(result.patch_output, b'test')

    def test_apply_patch_with_p(self):
        """Testing MercurialClient.apply_patch with p="""
        client = self.build_client()

        self.spy_on(run_process_exec,
                    op=kgb.SpyOpReturn((0, b'test', b'')))

        result = client.apply_patch(patch_file='test.diff',
                                    p='1',
                                    base_path='',
                                    base_dir='')

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', '-p', '1', 'test.diff', '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])

        self.assertTrue(result.applied)
        self.assertFalse(result.has_conflicts)
        self.assertEqual(result.conflicting_files, [])
        self.assertEqual(result.patch_output, b'test')

    def test_apply_patch_with_error(self):
        """Testing MercurialClient.apply_patch with error"""
        client = self.build_client()

        # Make sure we call this first, since we don't want the repository
        # lookups to fail in apply_patch() due to our spy.
        client.get_repository_info()

        self.spy_on(run_process_exec,
                    op=kgb.SpyOpReturn((1, b'bad', b'')))

        result = client.apply_patch(patch_file='test.diff',
                                    p=None,
                                    base_path='',
                                    base_dir='')

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', 'test.diff', '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])

        self.assertFalse(result.applied)
        self.assertFalse(result.has_conflicts)
        self.assertEqual(result.conflicting_files, [])
        self.assertEqual(result.patch_output, b'bad')

    def test_get_file_content(self) -> None:
        """Testing MercurialClient.get_file_content"""
        client = self.build_client()

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='delete and modify stuff')

        content = client.get_file_content(
            filename='foo.txt',
            revision=self._hg_get_tip())

        self.assertEqual(content, FOO1)

    def test_get_file_content_invalid_revision(self) -> None:
        """Testing MercurialClient.get_file_content with an invalid revision
        """
        client = self.build_client()

        with self.assertRaises(SCMError):
            client.get_file_content(
                filename='foo.txt',
                revision='aoeu')

    def test_get_file_content_invalid_filename(self) -> None:
        """Testing MercurialClient.get_file_content with an invalid filename
        """
        client = self.build_client()

        with self.assertRaises(SCMError):
            client.get_file_content(
                filename='unknown',
                revision='1')

    def test_get_file_size(self) -> None:
        """Testing MercurialClient.get_file_size"""
        client = self.build_client()

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO1,
                                msg='delete and modify stuff')

        size = client.get_file_size(
            filename='foo.txt',
            revision=self._hg_get_tip())

        self.assertEqual(size, len(FOO1))

    def test_get_file_size_invalid_revision(self) -> None:
        """Testing MercurialClient.get_file_size with an invalid revision"""
        client = self.build_client()

        with self.assertRaises(SCMError):
            client.get_file_size(
                filename='foo.txt',
                revision='aoeu')

    def test_get_file_size_invalid_filename(self) -> None:
        """Testing MercurialClient.get_file_size with an invalid filename"""
        client = self.build_client()

        with self.assertRaises(SCMError):
            client.get_file_size(
                filename='unknown',
                revision='1')


class MercurialSubversionClientTests(MercurialTestCase):
    """Unit tests for hgsubversion."""

    SVNSERVE_MAX_RETRIES = 12

    populate_default_clone = False

    _svnserve_pid = None
    _svn_temp_base_path = None
    _skip_reason = None

    @classmethod
    def setUpClass(cls):
        for exe in ('svnadmin', 'svnserve', 'svn'):
            if not is_exe_in_path(exe):
                cls._skip_reason = '%s is not available on the system.' % exe
                break
        else:
            has_hgsubversion = False

            try:
                output = (
                    run_process(
                        [
                            'hg', '--config', 'extensions.hgsubversion=',
                            'svn', '--help',
                        ],
                        ignore_errors=True)
                    .stdout
                    .read()
                )

                has_hgsubversion = \
                    not re.search('unknown command [\'"]svn[\'"]',
                                  output, re.I)
            except OSError:
                has_hgsubversion = False

            if not has_hgsubversion:
                cls._skip_reason = \
                    'hgsubversion is not available or cannot be used.'

        super().setUpClass()

        # Don't do any of the following expensive stuff if we know we're just
        # going to skip all the tests.
        if cls._skip_reason:
            return

        # Create the repository that we'll be populating and later cloning.
        temp_base_path = tempfile.mkdtemp(prefix='rbtools.')
        cls._svn_temp_base_path = temp_base_path

        svn_repo_path = os.path.join(temp_base_path, 'svnrepo')
        run_process(['svnadmin', 'create', svn_repo_path])

        # Fill it with content. First, though, we have to clone it.
        svn_checkout_path = os.path.join(temp_base_path, 'checkout.svn')
        run_process(['svn', 'checkout', 'file://%s' % svn_repo_path,
                     svn_checkout_path])
        os.chdir(svn_checkout_path)

        run_process(['svn', 'propset', 'reviewboard:url', cls.TESTSERVER,
                     svn_checkout_path])
        run_process(['svn', 'mkdir', 'trunk', 'branches', 'tags'])
        run_process(['svn', 'commit', '-m', 'Initial commit.'])
        os.chdir(os.path.join(svn_checkout_path, 'trunk'))

        for i, data in enumerate([FOO, FOO1, FOO2]):
            cls.svn_add_file_commit(filename='foo.txt',
                                    data=data,
                                    msg='Test commit %s' % i,
                                    add_file=(i == 0))

        # Launch svnserve so Mercurial can pull from it.
        svnserve_port = (os.environ.get('SVNSERVE_PORT') or
                         str(randint(30000, 40000)))

        pid_file = os.path.join(temp_base_path, 'svnserve.pid')

        run_process([
            'svnserve', '--single-thread', '--pid-file', pid_file, '-d',
            '--listen-port', svnserve_port, '-r', temp_base_path,
        ])

        for i in range(0, cls.SVNSERVE_MAX_RETRIES):
            try:
                cls._svnserve_pid = int(open(pid_file).read().strip())
            except (IOError, OSError):
                # Wait to see if svnserve has launched yet.
                time.sleep(0.25)

        if not cls._svnserve_pid:
            raise cls.failureException('Unable to launch svnserve on port %s'
                                       % svnserve_port)

        cls.svn_checkout_url = 'svn://127.0.0.1:%s/svnrepo' % svnserve_port

    @classmethod
    def tearDownClass(cls):
        if cls._svnserve_pid:
            os.kill(cls._svnserve_pid, 9)

        if cls._svn_temp_base_path:
            shutil.rmtree(cls._svn_temp_base_path, ignore_errors=True)

        super(MercurialSubversionClientTests, cls).tearDownClass()

    def setUp(self):
        super(MercurialSubversionClientTests, self).setUp()

        if self._skip_reason:
            raise unittest.SkipTest(self._skip_reason)

        home_dir = make_tempdir()
        self.set_user_home(home_dir)
        hgrc_path = os.path.join(home_dir, '.hgrc')

        # Make sure hgsubversion is enabled.
        #
        # This will modify the .hgrc in the temp home directory created for
        # these tests.
        #
        # The "hgsubversion =" tells Mercurial to check for hgsubversion in
        # the default PYTHONPATH
        with open(hgrc_path, 'w') as fp:
            fp.write('[extensions]\n')
            fp.write('hgsubversion =\n')

        try:
            self.clone_dir = os.path.join(home_dir, 'checkout.hg')
            self.run_hg(['clone', '--stream', self.svn_checkout_url,
                         self.clone_dir])
        except Exception as e:
            raise unittest.SkipTest(
                'Unable to clone Subversion repository: %s' % e)

        os.chdir(self.clone_dir)

    @classmethod
    def svn_add_file_commit(cls, filename, data, msg, add_file=True):
        with open(filename, 'wb') as fp:
            fp.write(data)

        if add_file:
            run_process(['svn', 'add', filename], ignore_errors=True)

        run_process(['svn', 'commit', '-m', msg])

    def test_get_repository_info(self):
        """Testing MercurialClient.get_repository_info with SVN"""
        client = self.build_client()
        ri = client.get_repository_info()

        self.assertEqual(client._type, 'svn')
        self.assertEqual(ri.base_path, '/trunk')
        self.assertEqual(ri.path, self.svn_checkout_url)

    def test_calculate_repository_info(self):
        """Testing MercurialClient._calculate_hgsubversion_repository_info
        with SVN determines repository and base paths
        """
        client = self.build_client()

        repo_info = client._calculate_hgsubversion_repository_info(
            'URL: svn+ssh://testuser@svn.example.net/repo/trunk\n'
            'Repository Root: svn+ssh://testuser@svn.example.net/repo\n'
            'Repository UUID: bfddb570-5023-0410-9bc8-bc1659bf7c01\n'
            'Revision: 9999\n'
            'Node Kind: directory\n'
            'Last Changed Author: user\n'
            'Last Changed Rev: 9999\n'
            'Last Changed Date: 2012-09-05 18:04:28 +0000 (Wed, 05 Sep 2012)'
        )

        self.assertEqual(repo_info.path, 'svn+ssh://svn.example.net/repo')
        self.assertEqual(repo_info.base_path, '/trunk')

    def test_scan_for_server_with_reviewboardrc(self):
        """Testing MercurialClient.scan_for_server with SVN and configured
        .reviewboardrc
        """
        client = self.build_client()

        with self.reviewboardrc({'REVIEWBOARD_URL': 'https://example.com/'}):
            client.config = load_config()
            ri = client.get_repository_info()

            self.assertEqual(client.scan_for_server(ri),
                             'https://example.com/')

    def test_scan_for_server_with_property(self):
        """Testing MercurialClient.scan_for_server with SVN and reviewboard:url
        property
        """
        client = self.build_client()
        ri = client.get_repository_info()

        self.assertEqual(client.scan_for_server(ri), self.TESTSERVER)

    def test_diff(self):
        """Testing MercurialClient.diff with SVN"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO4,
                                msg='edit 4')

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'diff': '',
                'parent_diff': None,
            })

    def test_diff_with_multiple_commits(self):
        """Testing MercurialClient.diff with SVN and multiple commits"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO4,
                                msg='edit 4')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO5,
                                msg='edit 5')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO6,
                                msg='edit 6')

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'diff': '',
                'parent_diff': None,
            })

    def test_diff_with_revision(self):
        """Testing MercurialClient.diff with SVN and specific revision"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()

        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO4,
                                msg='edit 4',
                                branch='b')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO5,
                                msg='edit 5',
                                branch='b')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO6,
                                msg='edit 6',
                                branch='b')
        self.hg_add_file_commit(filename='foo.txt',
                                data=FOO4,
                                msg='edit 7',
                                branch='b')

        revisions = client.parse_revision_spec(['3'])

        self.assertEqual(
            client.diff(revisions),
            {
                'diff': '',
                'parent_diff': None,
            })


class MercurialPatcherTests(MercurialTestCase):
    """Unit tests for MercurialPatcher.

    Version Added:
        5.1
    """

    def test_patch_git(self) -> None:
        """Testing MercurialPatcher.patch with Git-style diff"""
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                    % {
                        b'base_commit_id': tip.encode('utf-8'),
                    }
                )),
            ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', 'rb') as fp:
            self.assertEqual(fp.read(), FOO1)

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                 '--partial', tempfiles[0], '--config',
                 f'extensions.rbtoolsnormalize={hgext_path}',
            ])

        # There should not be a new commit, but should have pending changes.
        self.assertEqual(self._hg_get_tip(), f'{tip}+')

    def test_patch_hg(self) -> None:
        """Testing MercurialPatcher.patch with Hg-style diff"""
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff -r %(base_commit_id)s -r 001a1c12e834 foo.txt\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                    % {
                        b'base_commit_id': tip.encode('utf-8'),
                    }
                )),
            ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', 'rb') as fp:
            self.assertEqual(fp.read(), FOO1)

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', tempfiles[0], '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])

        # There should not be a new commit, but should have pending changes.
        self.assertEqual(self._hg_get_tip(), f'{tip}+')

    def test_patch_git_with_prefix_level(self) -> None:
        """Testing MercurialPatcher.patch with Git-style diff and
        Patch.prefix_level=
        """
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(
                    prefix_level=3,
                    content=(
                        b'# HG changeset patch\n'
                        b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
                        b'# Parent  %(base_commit_id)s\n'
                        b'diff --git a/b/c/foo.txt b/b/c/foo.txt\n'
                        b'--- a/b/c/foo.txt\n'
                        b'+++ b/b/c/foo.txt\n'
                        b'@@ -6,7 +6,4 @@\n'
                        b' inferretque deos Latio, genus unde Latinum,\n'
                        b' Albanique patres, atque altae moenia Romae.\n'
                        b' Musa, mihi causas memora, quo numine laeso,\n'
                        b'-quidve dolens, regina deum tot volvere casus\n'
                        b'-insignem pietate virum, tot adire labores\n'
                        b'-impulerit. Tantaene animis caelestibus irae?\n'
                        b' \n'
                        % {
                            b'base_commit_id': tip.encode('utf-8'),
                        }
                    )),
            ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', 'rb') as fp:
            self.assertEqual(fp.read(), FOO1)

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', '-p', '3', tempfiles[0], '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])

        # There should not be a new commit, but should have pending changes.
        self.assertEqual(self._hg_get_tip(), f'{tip}+')

    def test_patch_hg_with_prefix_level(self) -> None:
        """Testing MercurialPatcher.patch with Hg-style diff and
        Patch.prefix_level=
        """
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(
                    prefix_level=3,
                    content=(
                        b'# HG changeset patch\n'
                        b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
                        b'# Parent  %(base_commit_id)s\n'
                        b'diff -r %(base_commit_id)s -r 001a1c12e834'
                        b' b/c/foo.txt\n'
                        b'--- a/b/c/foo.txt\n'
                        b'+++ b/b/c/foo.txt\n'
                        b'@@ -6,7 +6,4 @@\n'
                        b' inferretque deos Latio, genus unde Latinum,\n'
                        b' Albanique patres, atque altae moenia Romae.\n'
                        b' Musa, mihi causas memora, quo numine laeso,\n'
                        b'-quidve dolens, regina deum tot volvere casus\n'
                        b'-insignem pietate virum, tot adire labores\n'
                        b'-impulerit. Tantaene animis caelestibus irae?\n'
                        b' \n'
                        % {
                            b'base_commit_id': tip.encode('utf-8'),
                        }
                    )),
            ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', 'rb') as fp:
            self.assertEqual(fp.read(), FOO1)

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', '-p', '3', tempfiles[0], '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])

        # There should not be a new commit, but should have pending changes.
        self.assertEqual(self._hg_get_tip(), f'{tip}+')

    def test_patch_with_revert(self) -> None:
        """Testing MercurialPatcher.patch with revert=True"""
        client = self.build_client()
        repository_info = client.get_repository_info()

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)

        patcher = client.get_patcher(
            repository_info=repository_info,
            revert=True,
            patches=[
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                    % {
                        b'base_commit_id': tip.encode('utf-8'),
                    }
                )),
            ])

        message = 'Mercurial does not support reverting patches.'

        with self.assertRaisesMessage(ApplyPatchError, message):
            list(patcher.patch())

        self.assertSpyNotCalled(run_process_exec)

        # There should not be a new commit.
        self.assertEqual(self._hg_get_tip(), tip)

    def test_patch_with_commit(self) -> None:
        """Testing MercurialPatcher.patch with committing"""
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        tip = self._hg_get_tip()
        num_commits = self.hg_get_num_commits()

        self.spy_on(run_process_exec)

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                    % {
                        b'base_commit_id': tip.encode('utf-8'),
                    }
                )),
            ])
        patcher.prepare_for_commit(
            default_author=PatchAuthor(full_name='Test User',
                                       email='test@example.com'),
            default_message='Test message')

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', 'rb') as fp:
            self.assertEqual(fp.read(), FOO1)

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', tempfiles[0], '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])

        # There should be a new commit.
        self.assertEqual(self.hg_get_num_commits(), num_commits + 1)

    def test_patch_with_multiple_patches(self) -> None:
        """Testing MercurialPatcher.patch with multiple patches"""
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(2)

        commit1 = self._hg_get_tip()
        commit2 = b'61ee8b5601a84d5154387578466c8998848ba089'
        commit3 = b'b9af6489f6f2004ad11b82c6057f7007e3c35372'

        self.spy_on(run_process_exec, op=kgb.SpyOpMatchInOrder([
            {
                'args': ([
                    'hg', 'import', '--config', 'patch.eol=auto',
                    '--no-commit', '--partial', tempfiles[0],
                    tempfiles[1], '--config',
                    f'extensions.rbtoolsnormalize={hgext_path}',
                ],)
            },
        ]))

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID %(commit_id)s\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                    % {
                        b'base_commit_id': commit1.encode('utf-8'),
                        b'commit_id': commit2,
                    }
                )),
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID %(commit_id)s\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -1,4 +1,6 @@\n'
                    b' ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' Italiam, fato profugus, Laviniaque venit\n'
                    b' litora, multum ille et terris iactatus et alto\n'
                    b' vi superum saevae memorem Iunonis ob iram;\n'
                    % {
                        b'base_commit_id': commit2,
                        b'commit_id': commit3,
                    }
                )),
            ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNone(result.patch)

        with open('foo.txt', 'rb') as fp:
            self.assertEqual(fp.read(), FOO2)

        self.assertSpyCallCount(run_process_exec, 1)

        run_process_exec.unspy()

        # There should not be a new commit.
        self.assertEqual(self._hg_get_tip(), f'{commit1}+')

    def test_patch_with_multiple_patches_and_commit(self) -> None:
        """Testing MercurialPatcher.patch with multiple patches and commit"""
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(2)

        num_commits = self.hg_get_num_commits()
        commit1 = self._hg_get_tip()
        commit2 = b'61ee8b5601a84d5154387578466c8998848ba089'
        commit3 = b'b9af6489f6f2004ad11b82c6057f7007e3c35372'

        self.spy_on(run_process_exec, op=kgb.SpyOpMatchInOrder([
            {
                'args': ([
                    'hg', 'import', '--config', 'patch.eol=auto',
                    '--no-commit', '--partial', tempfiles[0],
                    '--config', f'extensions.rbtoolsnormalize={hgext_path}',
                ],)
            },
            {
                'args': ([
                    'hg', 'commit', '-m', '[1/2] Test message',
                    '-u', 'Test User <test@example.com>',
                    '--config', f'extensions.rbtoolsnormalize={hgext_path}',
                ],)
            },
            {
                'args': ([
                    'hg', 'import', '--config', 'patch.eol=auto',
                    '--no-commit', '--partial', tempfiles[1],
                    '--config', f'extensions.rbtoolsnormalize={hgext_path}',
                ],)
            },
            {
                'args': ([
                    'hg', 'commit', '-m', '[2/2] Test message',
                    '-u', 'Test User <test@example.com>',
                    '--config', f'extensions.rbtoolsnormalize={hgext_path}',
                ],)
            },
        ]))

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID %(commit_id)s\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                    % {
                        b'base_commit_id': commit1.encode('utf-8'),
                        b'commit_id': commit2,
                    }
                )),
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID %(commit_id)s\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -1,4 +1,6 @@\n'
                    b' ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' Italiam, fato profugus, Laviniaque venit\n'
                    b' litora, multum ille et terris iactatus et alto\n'
                    b' vi superum saevae memorem Iunonis ob iram;\n'
                    % {
                        b'base_commit_id': commit2,
                        b'commit_id': commit3,
                    }
                )),
            ])
        patcher.prepare_for_commit(
            default_author=PatchAuthor(full_name='Test User',
                                       email='test@example.com'),
            default_message='Test message')

        results = list(patcher.patch())
        self.assertEqual(len(results), 2)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        result = results[1]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', 'rb') as fp:
            self.assertEqual(fp.read(), FOO2)

        self.assertSpyCallCount(run_process_exec, 4)

        run_process_exec.unspy()

        # There should be a new commit.
        self.assertEqual(self.hg_get_num_commits(), num_commits + 2)

    def test_patch_git_with_conflicts(self) -> None:
        """Testing MercurialPatcher.patch with Git-style diff and conflicts"""
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere CASUS\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                    % {
                        b'base_commit_id': tip.encode('utf-8'),
                    }
                )),
            ])

        message = (
            'Could not apply patch 1 of 1. The patch may be invalid, or '
            'there may be conflicts that could not be resolved.'
        )

        with self.assertRaisesMessage(ApplyPatchError, message) as ctx:
            list(patcher.patch())

        result = ctx.exception.failed_patch_result
        self.assertFalse(result.success)
        self.assertFalse(result.applied)
        self.assertTrue(result.has_conflicts)
        self.assertIsNotNone(result.patch)
        self.assertEqual(result.conflicting_files, ['foo.txt'])
        self.assertEqual(
            result.patch_output,
            b'applying %s\n'
            b'patching file foo.txt\n'
            b'Hunk #1 FAILED at 5\n'
            b'1 out of 1 hunks FAILED -- saving rejects to file foo.txt.rej\n'
            b'patch applied partially\n'
            b'(fix the .rej files and run `hg commit --amend`)\n'
            % tempfiles[0].encode('utf-8'))

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', tempfiles[0], '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])

        # There should not be a new commit.
        self.assertEqual(self._hg_get_tip(), tip)

    def test_patch_hg_with_conflicts(self) -> None:
        """Testing MercurialPatcher.patch with Hg-style diff and conflicts"""
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff -r %(base_commit_id)s -r 001a1c12e834 foo.txt\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere CASUS\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                    % {
                        b'base_commit_id': tip.encode('utf-8'),
                    }
                )),
            ])

        message = (
            'Could not apply patch 1 of 1. The patch may be invalid, or '
            'there may be conflicts that could not be resolved.'
        )

        with self.assertRaisesMessage(ApplyPatchError, message) as ctx:
            list(patcher.patch())

        result = ctx.exception.failed_patch_result
        self.assertFalse(result.success)
        self.assertFalse(result.applied)
        self.assertTrue(result.has_conflicts)
        self.assertIsNotNone(result.patch)
        self.assertEqual(result.conflicting_files, ['foo.txt'])
        self.assertEqual(
            result.patch_output,
            b'applying %s\n'
            b'patching file foo.txt\n'
            b'Hunk #1 FAILED at 5\n'
            b'1 out of 1 hunks FAILED -- saving rejects to file foo.txt.rej\n'
            b'patch applied partially\n'
            b'(fix the .rej files and run `hg commit --amend`)\n'
            % tempfiles[0].encode('utf-8'))

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', tempfiles[0], '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])

        # There should not be a new commit.
        self.assertEqual(self._hg_get_tip(), tip)

    # NOTE: There's a test for Git-style but not Hg-style because Hg-style
    #       diffs don't convey added/deleted files.
    def test_patch_git_with_empty_files(self) -> None:
        """Testing MercurialPatcher.patch with Git-style diff and empty files
        """
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        self.hg_add_file_commit(filename='empty1',
                                data=b'',
                                msg='Add an empty file.')

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff --git a/empty1 b/empty1\n'
                    b'deleted file mode 100644\n'
                    b'diff --git a/newfile b/newfile\n'
                    b'new file mode 100644\n'
                    % {
                        b'base_commit_id': tip.encode('utf-8'),
                    }
                )),
            ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        self.assertFalse(os.path.exists('empty1'))
        self.assertTrue(os.path.exists('newfile'))

        with open('newfile', 'rb') as fp:
            self.assertEqual(fp.read(), b'')

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', tempfiles[0], '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])

        # There should not be a new commit, but should have pending changes.
        self.assertEqual(self._hg_get_tip(), f'{tip}+')

    def test_binary_file_add(self) -> None:
        """Testing MercurialPatcher with an added binary file"""
        client = self.build_client()
        repository_info = client.get_repository_info()

        test_content = b'Binary file content'
        test_path = 'new_binary_file.bin'

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 123,
                'absolute_url': 'https://example.com/r/1/file/123/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/123/'
        )

        binary_file = self.make_binary_file_patch(
            old_path=None,
            new_path=test_path,
            status='added',
            file_attachment=attachment,
            content=test_content,
        )

        tip = self._hg_get_tip()

        patch_content = (
            b'# HG changeset patch\n'
            b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
            b'# Parent %(base_commit_id)s\n'
            b'diff -r %(base_commit_id)s -r 001a1c12e834 %(path)s\n'
            b'Binary file %(path)s has changed\n'
            % {
                b'base_commit_id': tip.encode(),
                b'path': test_path.encode(),
            }
        )

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=patch_content, binary_files=[binary_file]),
            ])

        self.spy_on(client._execute)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], test_path)

        self.assertTrue(os.path.exists(test_path))

        with open(test_path, 'rb') as f:
            self.assertEqual(f.read(), test_content)

        self.assertSpyCalledWith(
            client._execute, ['hg', 'add', test_path])

    def test_binary_file_add_in_subdirectory(self) -> None:
        """Testing MercurialPatcher with an added binary file in subdirectory
        """
        client = self.build_client()
        repository_info = client.get_repository_info()

        test_content = b'Binary data in subdirectory'
        test_path = 'subdir/nested_binary_file.bin'

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 124,
                'absolute_url': 'https://example.com/r/1/file/124/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/124/'
        )

        binary_file = self.make_binary_file_patch(
            old_path=None,
            new_path=test_path,
            status='added',
            file_attachment=attachment,
            content=test_content,
        )

        tip = self._hg_get_tip()

        patch_content = (
            b'# HG changeset patch\n'
            b'# Node ID 002b2d23f945294b06745701fc9bc76db3822105\n'
            b'# Parent %(base_commit_id)s\n'
            b'diff -r %(base_commit_id)s -r 002b2d23f945 %(path)s\n'
            b'Binary file %(path)s has changed\n'
            % {
                b'base_commit_id': tip.encode(),
                b'path': test_path.encode(),
            }
        )

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=patch_content, binary_files=[binary_file]),
            ])

        self.spy_on(client._execute)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], test_path)

        self.assertTrue(os.path.exists(test_path))
        self.assertTrue(os.path.isdir('subdir'))

        with open(test_path, 'rb') as f:
            self.assertEqual(f.read(), test_content)

        self.assertSpyCalledWith(
            client._execute, ['hg', 'add', test_path])

    def test_binary_file_move(self) -> None:
        """Testing MercurialPatcher with a moved binary file"""
        client = self.build_client()
        repository_info = client.get_repository_info()

        old_content = b'Original binary content'
        new_content = b'Updated binary content'
        old_path = 'original.bin'
        new_path = 'renamed.bin'

        self.hg_add_file_commit(old_path, old_content)

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 125,
                'absolute_url': 'https://example.com/r/1/file/125/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/125/'
        )

        binary_file = self.make_binary_file_patch(
            old_path=old_path,
            new_path=new_path,
            status='moved',
            file_attachment=attachment,
            content=new_content,
        )

        tip = self._hg_get_tip()

        patch_content = (
            b'# HG changeset patch\n'
            b'# Node ID 003c3e34g056305c17856812gd0cd87ec4933216\n'
            b'# Parent %(base_commit_id)s\n'
            b'diff -r %(base_commit_id)s -r 003c3e34g056 %(old_path)s\n'
            b'rename from %(old_path)s\n'
            b'rename to %(new_path)s\n'
            b'Binary file %(new_path)s has changed\n'
            % {
                b'base_commit_id': tip.encode(),
                b'old_path': old_path.encode(),
                b'new_path': new_path.encode(),
            }
        )

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=patch_content, binary_files=[binary_file]),
            ])

        self.spy_on(client._execute)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], new_path)

        self.assertFalse(os.path.exists(old_path))

        self.assertTrue(os.path.exists(new_path))
        with open(new_path, 'rb') as f:
            self.assertEqual(f.read(), new_content)

        self.assertSpyCalledWith(
            client._execute, ['hg', 'rename', old_path, new_path])

    def test_binary_file_move_with_subdirectory(self) -> None:
        """Testing MercurialPatcher with a moved binary file and a new
        subdirectory
        """
        client = self.build_client()
        repository_info = client.get_repository_info()

        old_content = b'File to be moved'
        new_content = b'Updated file in subdirectory'
        old_path = 'root_file.bin'
        new_path = 'subdir/moved_file.bin'

        self.hg_add_file_commit(old_path, old_content)

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 126,
                'absolute_url': 'https://example.com/r/1/file/126/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/126/'
        )

        binary_file = self.make_binary_file_patch(
            old_path=old_path,
            new_path=new_path,
            status='moved',
            file_attachment=attachment,
            content=new_content,
        )

        tip = self._hg_get_tip()

        patch_content = (
            b'# HG changeset patch\n'
            b'# Node ID 004d4f45h167416d28967923he1de98fd5044327\n'
            b'# Parent %(base_commit_id)s\n'
            b'diff -r %(base_commit_id)s -r 004d4f45h167 %(old_path)s\n'
            b'rename from %(old_path)s\n'
            b'rename to %(new_path)s\n'
            b'Binary file %(new_path)s has changed\n'
            % {
                b'base_commit_id': tip.encode(),
                b'old_path': old_path.encode(),
                b'new_path': new_path.encode(),
            }
        )

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=patch_content, binary_files=[binary_file]),
            ])

        self.spy_on(client._execute)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], new_path)

        self.assertFalse(os.path.exists(old_path))

        self.assertTrue(os.path.isdir('subdir'))
        self.assertTrue(os.path.exists(new_path))
        with open(new_path, 'rb') as f:
            self.assertEqual(f.read(), new_content)

        self.assertSpyCalledWith(
            client._execute, ['hg', 'rename', old_path, new_path])

    def test_binary_file_remove(self) -> None:
        """Testing MercurialPatcher with a removed binary file"""
        client = self.build_client()
        repository_info = client.get_repository_info()

        file_content = b'Content to be removed'
        file_path = 'file_to_remove.bin'

        self.hg_add_file_commit(file_path, file_content)

        binary_file = BinaryFilePatch(
            old_path=file_path,
            new_path=None,
            status='deleted',
            file_attachment=None,
        )

        tip = self._hg_get_tip()

        patch_content = (
            b'# HG changeset patch\n'
            b'# Node ID 005e5g56i278527e39078034if2ef09ge6155438\n'
            b'# Parent %(base_commit_id)s\n'
            b'diff -r %(base_commit_id)s -r 005e5g56i278 %(file_path)s\n'
            b'Binary file %(file_path)s has changed\n'
            % {
                b'base_commit_id': tip.encode(),
                b'file_path': file_path.encode(),
            }
        )

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=patch_content, binary_files=[binary_file]),
            ])

        self.spy_on(client._execute)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], file_path)

        self.assertFalse(os.path.exists(file_path))

        self.assertSpyCalledWith(
            client._execute, ['hg', 'remove', file_path])

    def test_binary_files_with_multiple_patches(self) -> None:
        """Testing MercurialPatcher.patch with multiple patches containing
        binary files
        """
        client = self.build_client()
        repository_info = client.get_repository_info()

        # First binary file - addition
        content1 = b'First binary content'
        path1 = 'first.bin'

        attachment1 = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 130,
                'absolute_url': 'https://example.com/r/1/file/130/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/130/'
        )

        binary_file1 = self.make_binary_file_patch(
            old_path=None,
            new_path=path1,
            status='added',
            file_attachment=attachment1,
            content=content1,
        )

        # Second binary file - addition
        content2 = b'Second binary content'
        path2 = 'second.bin'

        attachment2 = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 131,
                'absolute_url': 'https://example.com/r/1/file/131/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/131/'
        )

        binary_file2 = self.make_binary_file_patch(
            old_path=None,
            new_path=path2,
            status='added',
            file_attachment=attachment2,
            content=content2,
        )

        tip = self._hg_get_tip()
        commit1_id = b'007h7i78k490749g51290256kh4hg21ig8377660'
        commit2_id = b'008i8j89l501850h62301367li5ih32jh9488771'

        patch_content1 = (
            b'# HG changeset patch\n'
            b'# Node ID %(commit_id)s\n'
            b'# Parent %(base_commit_id)s\n'
            b'diff -r %(base_commit_id)s -r 007h7i78k490 %(path)s\n'
            b'Binary file %(path)s has changed\n'
            % {
                b'base_commit_id': tip.encode(),
                b'commit_id': commit1_id,
                b'path': path1.encode(),
            }
        )

        patch_content2 = (
            b'# HG changeset patch\n'
            b'# Node ID %(commit_id)s\n'
            b'# Parent %(base_commit_id)s\n'
            b'diff -r %(base_commit_id)s -r 008i8j89l501 %(path)s\n'
            b'Binary file %(path)s has changed\n'
            % {
                b'base_commit_id': commit1_id,
                b'commit_id': commit2_id,
                b'path': path2.encode(),
            }
        )

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=patch_content1, binary_files=[binary_file1]),
                Patch(content=patch_content2, binary_files=[binary_file2]),
            ])

        self.spy_on(client._execute)

        results = list(patcher.patch())

        # When not committing, there's only one patch result.
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 2)
        self.assertEqual(result.binary_applied[0], path1)
        self.assertEqual(result.binary_applied[1], path2)

        # Verify files were created with correct content
        self.assertTrue(os.path.exists(path1))

        with open(path1, 'rb') as f:
            self.assertEqual(f.read(), content1)

        self.assertTrue(os.path.exists(path2))

        with open(path2, 'rb') as f:
            self.assertEqual(f.read(), content2)

        # Verify the hg add commands were called for both files
        self.assertSpyCalledWith(client._execute, ['hg', 'add', path1])
        self.assertSpyCalledWith(client._execute, ['hg', 'add', path2])

    def test_binary_files_with_multiple_patches_and_commit(self) -> None:
        """Testing MercurialPatcher.patch with multiple patches containing
        binary files and commit
        """
        client = self.build_client()
        repository_info = client.get_repository_info()

        # First binary file - addition
        content1 = b'First binary content for commit'
        path1 = 'committed_first.bin'

        attachment1 = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 132,
                'absolute_url': 'https://example.com/r/1/file/132/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/132/'
        )

        binary_file1 = self.make_binary_file_patch(
            old_path=None,
            new_path=path1,
            status='added',
            file_attachment=attachment1,
            content=content1,
        )

        # Second binary file - addition
        content2 = b'Second binary content for commit'
        path2 = 'committed_second.bin'

        attachment2 = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 133,
                'absolute_url': 'https://example.com/r/1/file/133/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/133/'
        )

        binary_file2 = self.make_binary_file_patch(
            old_path=None,
            new_path=path2,
            status='added',
            file_attachment=attachment2,
            content=content2,
        )

        tip = self._hg_get_tip()
        commit1_id = b'009j9k90m612961i73412478mj6jh43ki0599882'
        commit2_id = b'010k0l01n723072j84523589nk7ki54lj1600993'

        patch_content1 = (
            b'# HG changeset patch\n'
            b'# Node ID %(commit_id)s\n'
            b'# Parent %(base_commit_id)s\n'
            b'First binary patch commit\n'
            b'\n'
            b'diff -r %(base_commit_id)s -r 009j9k90m612 %(path)s\n'
            b'Binary file %(path)s has changed\n'
            % {
                b'base_commit_id': tip.encode(),
                b'commit_id': commit1_id,
                b'path': path1.encode(),
            }
        )

        patch_content2 = (
            b'# HG changeset patch\n'
            b'# Node ID %(commit_id)s\n'
            b'# Parent %(base_commit_id)s\n'
            b'Second binary patch commit\n'
            b'\n'
            b'diff -r %(base_commit_id)s -r 010k0l01n723 %(path)s\n'
            b'Binary file %(path)s has changed\n'
            % {
                b'base_commit_id': commit1_id,
                b'commit_id': commit2_id,
                b'path': path2.encode(),
            }
        )

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=patch_content1, binary_files=[binary_file1]),
                Patch(content=patch_content2, binary_files=[binary_file2]),
            ])
        patcher.prepare_for_commit(
            default_author=PatchAuthor(full_name='Test User',
                                       email='test@example.com'),
            default_message='Test message')

        self.spy_on(client._execute)

        results = list(patcher.patch())
        self.assertEqual(len(results), 2)

        result1 = results[0]
        self.assertTrue(result1.success)
        self.assertEqual(len(result1.binary_applied), 1)
        self.assertEqual(result1.binary_applied[0], path1)

        result2 = results[1]
        self.assertTrue(result2.success)
        self.assertEqual(len(result2.binary_applied), 1)
        self.assertEqual(result2.binary_applied[0], path2)

        self.assertTrue(os.path.exists(path1))
        with open(path1, 'rb') as f:
            self.assertEqual(f.read(), content1)

        self.assertTrue(os.path.exists(path2))
        with open(path2, 'rb') as f:
            self.assertEqual(f.read(), content2)

        self.assertSpyCalledWith(client._execute, ['hg', 'add', path1])
        self.assertSpyCalledWith(client._execute, ['hg', 'add', path2])

        final_tip = self._hg_get_tip()
        self.assertNotEqual(tip, final_tip)

    def test_patch_with_regular_and_empty_files(self) -> None:
        """Testing MercurialPatcher.patch with regular and empty files."""
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        self.hg_add_file_commit(filename='empty_file',
                                data=b'',
                                msg='Add an empty file.')

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)

        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[
                Patch(content=(
                    b'# HG changeset patch\n'
                    b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
                    b'# Parent  %(base_commit_id)s\n'
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,8 @@\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'+New line added\n'
                    b' quidve dolens, regina deum tot volvere casus\n'
                    b' insignem pietate virum, tot adire labores\n'
                    b' impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                    b'diff --git a/empty_file b/empty_file\n'
                    b'deleted file mode 100644\n'
                    b'diff --git a/new_empty_file b/new_empty_file\n'
                    b'new file mode 100644\n'
                    % {
                        b'base_commit_id': tip.encode('utf-8'),
                    }
                )),
            ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', 'rb') as fp:
            content = fp.read()
            self.assertIn(b'New line added', content)

        self.assertFalse(os.path.exists('empty_file'))
        self.assertTrue(os.path.exists('new_empty_file'))

        with open('new_empty_file', 'rb') as fp:
            self.assertEqual(fp.read(), b'')

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', tempfiles[0], '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])

    def test_patch_with_regular_and_binary_files(self) -> None:
        """Testing MercurialPatcher.patch with regular and binary files."""
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        test_content1 = b'Binary file content 1'
        test_content2 = b'Binary file content 2'

        attachment1 = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 201,
                'absolute_url': 'https://example.com/r/1/file/201/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/201/'
        )

        attachment2 = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 202,
                'absolute_url': 'https://example.com/r/1/file/202/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/202/'
        )

        binary_file1 = self.make_binary_file_patch(
            old_path=None,
            new_path='new_binary.bin',
            status='added',
            file_attachment=attachment1,
            content=test_content1,
        )

        binary_file2 = self.make_binary_file_patch(
            old_path='bar.txt',
            new_path='bar.txt',
            status='modified',
            file_attachment=attachment2,
            content=test_content2,
        )

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)
        self.spy_on(client._execute)

        patch_content = (
            b'# HG changeset patch\n'
            b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
            b'# Parent  %(base_commit_id)s\n'
            b'diff --git a/foo.txt b/foo.txt\n'
            b'--- a/foo.txt\n'
            b'+++ b/foo.txt\n'
            b'@@ -6,7 +6,8 @@\n'
            b' inferretque deos Latio, genus unde Latinum,\n'
            b' Albanique patres, atque altae moenia Romae.\n'
            b' Musa, mihi causas memora, quo numine laeso,\n'
            b'+New line added\n'
            b' quidve dolens, regina deum tot volvere casus\n'
            b' insignem pietate virum, tot adire labores\n'
            b' impulerit. Tantaene animis caelestibus irae?\n'
            b' \n'
            b'diff -r %(base_commit_id)s -r 001a1c12e834 new_binary.bin\n'
            b'Binary file new_binary.bin has changed\n'
            b'diff -r %(base_commit_id)s -r 001a1c12e834 bar.txt\n'
            b'Binary file bar.txt has changed\n'
            % {
                b'base_commit_id': tip.encode('utf-8'),
            }
        )

        patch = Patch(content=patch_content,
                      binary_files=[binary_file1, binary_file2])
        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[patch])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 2)

        with open('foo.txt', 'rb') as fp:
            content = fp.read()
            self.assertIn(b'New line added', content)

        self.assertTrue(os.path.exists('new_binary.bin'))
        self.assertTrue(os.path.exists('bar.txt'))

        with open('new_binary.bin', 'rb') as f:
            self.assertEqual(f.read(), test_content1)

        with open('bar.txt', 'rb') as f:
            self.assertEqual(f.read(), test_content2)

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', tempfiles[0], '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])
        self.assertSpyCalledWith(client._execute,
                                 ['hg', 'add', 'new_binary.bin'])

    def test_patch_with_empty_and_binary_files(self) -> None:
        """Testing MercurialPatcher.patch with empty and binary files."""
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        self.hg_add_file_commit(filename='empty_file',
                                data=b'',
                                msg='Add an empty file.')

        test_content = b'Binary file content'

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 301,
                'absolute_url': 'https://example.com/r/1/file/301/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/301/'
        )

        binary_file = self.make_binary_file_patch(
            old_path=None,
            new_path='new_binary.bin',
            status='added',
            file_attachment=attachment,
            content=test_content,
        )

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)
        self.spy_on(client._execute)

        patch_content = (
            b'# HG changeset patch\n'
            b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
            b'# Parent  %(base_commit_id)s\n'
            b'diff --git a/empty_file b/empty_file\n'
            b'deleted file mode 100644\n'
            b'diff --git a/new_empty_file b/new_empty_file\n'
            b'new file mode 100644\n'
            b'diff -r %(base_commit_id)s -r 001a1c12e834 new_binary.bin\n'
            b'Binary file new_binary.bin has changed\n'
            % {
                b'base_commit_id': tip.encode('utf-8'),
            }
        )

        patch = Patch(content=patch_content, binary_files=[binary_file])
        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[patch])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)

        self.assertFalse(os.path.exists('empty_file'))
        self.assertTrue(os.path.exists('new_empty_file'))

        with open('new_empty_file', 'rb') as fp:
            self.assertEqual(fp.read(), b'')

        self.assertTrue(os.path.exists('new_binary.bin'))

        with open('new_binary.bin', 'rb') as f:
            self.assertEqual(f.read(), test_content)

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', tempfiles[0], '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])
        self.assertSpyCalledWith(client._execute,
                                 ['hg', 'add', 'new_binary.bin'])

    def test_patch_with_mixed_file_types(self) -> None:
        """Testing MercurialPatcher.patch with regular, empty, and binary
        files.
        """
        client = self.build_client()
        repository_info = client.get_repository_info()
        tempfiles = self.precreate_tempfiles(1)

        self.hg_add_file_commit(filename='empty_file',
                                data=b'',
                                msg='Add an empty file.')

        test_content1 = b'New binary content'
        test_content2 = b'Modified binary content'

        attachment1 = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 401,
                'absolute_url': 'https://example.com/r/1/file/401/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/401/'
        )

        attachment2 = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 402,
                'absolute_url': 'https://example.com/r/1/file/402/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/402/'
        )

        binary_file1 = self.make_binary_file_patch(
            old_path=None,
            new_path='new_binary.bin',
            status='added',
            file_attachment=attachment1,
            content=test_content1,
        )

        binary_file2 = self.make_binary_file_patch(
            old_path='bar.txt',
            new_path='bar.txt',
            status='modified',
            file_attachment=attachment2,
            content=test_content2,
        )

        tip = self._hg_get_tip()

        self.spy_on(run_process_exec)
        self.spy_on(client._execute)

        patch_content = (
            b'# HG changeset patch\n'
            b'# Node ID 001a1c12e834183a95634690eb8ab65ca2711094\n'
            b'# Parent  %(base_commit_id)s\n'
            b'diff --git a/foo.txt b/foo.txt\n'
            b'--- a/foo.txt\n'
            b'+++ b/foo.txt\n'
            b'@@ -6,7 +6,8 @@\n'
            b' inferretque deos Latio, genus unde Latinum,\n'
            b' Albanique patres, atque altae moenia Romae.\n'
            b' Musa, mihi causas memora, quo numine laeso,\n'
            b'+New line added\n'
            b' quidve dolens, regina deum tot volvere casus\n'
            b' insignem pietate virum, tot adire labores\n'
            b' impulerit. Tantaene animis caelestibus irae?\n'
            b' \n'
            b'diff --git a/empty_file b/empty_file\n'
            b'deleted file mode 100644\n'
            b'diff --git a/new_empty_file b/new_empty_file\n'
            b'new file mode 100644\n'
            b'diff -r %(base_commit_id)s -r 001a1c12e834 new_binary.bin\n'
            b'Binary file new_binary.bin has changed\n'
            b'diff -r %(base_commit_id)s -r 001a1c12e834 bar.txt\n'
            b'Binary file bar.txt has changed\n'
            % {
                b'base_commit_id': tip.encode('utf-8'),
            }
        )

        patch = Patch(content=patch_content,
                      binary_files=[binary_file1, binary_file2])
        patcher = client.get_patcher(
            repository_info=repository_info,
            patches=[patch])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 2)

        with open('foo.txt', 'rb') as fp:
            content = fp.read()
            self.assertIn(b'New line added', content)

        self.assertFalse(os.path.exists('empty_file'))
        self.assertTrue(os.path.exists('new_empty_file'))

        with open('new_empty_file', 'rb') as fp:
            self.assertEqual(fp.read(), b'')

        self.assertTrue(os.path.exists('new_binary.bin'))
        self.assertTrue(os.path.exists('bar.txt'))

        with open('new_binary.bin', 'rb') as f:
            self.assertEqual(f.read(), test_content1)

        with open('bar.txt', 'rb') as f:
            self.assertEqual(f.read(), test_content2)

        self.assertSpyCalledWith(
            run_process_exec,
            [
                'hg', 'import', '--config', 'patch.eol=auto', '--no-commit',
                '--partial', tempfiles[0], '--config',
                f'extensions.rbtoolsnormalize={hgext_path}',
            ])
        self.assertSpyCalledWith(client._execute,
                                 ['hg', 'add', 'new_binary.bin'])
