"""Unit tests for GitClient."""

from __future__ import annotations

import os
import re
import unittest
from typing import ClassVar, TYPE_CHECKING

import kgb

from rbtools.api.resource import FileAttachmentItemResource
from rbtools.clients import RepositoryInfo
from rbtools.clients.errors import (CreateCommitError,
                                    MergeError,
                                    PushError,
                                    SCMClientDependencyError,
                                    SCMError,
                                    TooManyRevisionsError)
from rbtools.clients.git import GitClient, get_git_candidates
from rbtools.clients.tests import FOO1, FOO2, FOO3, FOO4, SCMClientTestCase
from rbtools.diffs.patches import BinaryFilePatch, Patch, PatchAuthor
from rbtools.testing.api.transport import URLMapTransport
from rbtools.utils.checks import check_install
from rbtools.utils.filesystem import is_exe_in_path
from rbtools.utils.process import (RunProcessResult,
                                   run_process,
                                   run_process_exec)

if TYPE_CHECKING:
    from collections.abc import Sequence


class BaseGitClientTests(SCMClientTestCase[GitClient]):
    """Base class for unit tests for GitClient.

    Version Added:
        4.0
    """

    default_scmclient_caps = {
        'scmtools': {
            'git': {
                'empty_files': True,
            },
        },
    }

    #: The SCMClient class to instantiate.
    scmclient_cls = GitClient

    #: The git executable to use.
    _git: ClassVar[str] = ''

    #: The top-level Git directory.
    git_dir: ClassVar[str]

    #: The clone directory.
    clone_dir: ClassVar[str]

    @classmethod
    def setup_checkout(
        cls,
        checkout_dir: str,
    ) -> str | None:
        """Populate a Git checkout.

        This will create a checkout of the sample Git repository stored
        in the :file:`testdata` directory, along with a child clone and a
        grandchild clone.

        Args:
            checkout_dir (unicode):
                The top-level directory in which the clones will be placed.

        Returns:
            str:
            The main checkout directory, or ``None`` if :command:`git` isn't
            in the path.
        """
        scmclient = GitClient()

        if not scmclient.has_dependencies():
            return None

        cls._git = scmclient.git

        cls.git_dir = os.path.join(cls.testdata_dir, 'git-repo')
        cls.clone_dir = checkout_dir

        os.mkdir(checkout_dir, 0o700)

        return checkout_dir

    @classmethod
    def _run_git(
        cls,
        command: Sequence[str],
    ) -> RunProcessResult:
        """Run git with the provided arguments.

        Args:
            command (list of str):
                The arguments to pass to :command:`git`.

        Returns:
            rbtools.utils.process.RunProcessResult:
            The result of the :py:func:`~rbtools.utils.process.run_process`
            call.
        """
        return run_process([cls._git, *command])

    @classmethod
    def _git_add_file_commit(
        cls,
        filename: str,
        data: bytes,
        msg: str,
    ) -> None:
        """Add a file to a git repository.

        Args:
            filename (str):
                The filename to write to.

            data (bytes):
                The content of the file to write.

            msg (str):
                The commit message to use.
        """
        with open(filename, 'wb') as f:
            f.write(data)

        cls._run_git(['add', filename])
        cls._run_git(['commit', '-m', msg])

    def setUp(self) -> None:
        """Set up the test case."""
        super().setUp()

        self.set_user_home(os.path.join(self.testdata_dir, 'homedir'))

    def _git_get_head(self) -> str:
        """Return the HEAD commit SHA.

        Returns:
            str:
            The HEAD commit SHA.
        """
        return (
            self._run_git(['rev-parse', 'HEAD'])
            .stdout
            .read()
            .strip()
        )

    def _git_get_num_commits(self) -> int:
        """Return the number of commits in the repository.

        Version Added:
            5.1

        Returns:
            int:
            The number of commits.
        """
        return len(
            self._run_git(['log', '--oneline'])
            .stdout
            .readlines()
        )


class GitClientTests(BaseGitClientTests):
    """Unit tests for GitClient."""

    TESTSERVER: ClassVar[str] = 'http://127.0.0.1:8080'
    AUTHOR: ClassVar[PatchAuthor] = PatchAuthor(full_name='name',
                                                email='email')

    #: The directory with the child git clone.
    child_clone_dir: ClassVar[str]

    #: The directory with the grandchild git clone.
    grandchild_clone_dir: ClassVar[str]

    #: The directory with the original git clone
    orig_clone_dir: ClassVar[str]

    @classmethod
    def setup_checkout(
        cls,
        checkout_dir: str,
    ) -> str | None:
        """Populate a Git checkout.

        This will create a checkout of the sample Git repository stored
        in the :file:`testdata` directory, along with a child clone and a
        grandchild clone.

        Args:
            checkout_dir (unicode):
                The top-level directory in which the clones will be placed.

        Returns:
            str:
            The main checkout directory, or ``None`` if :command:`git` isn't
            in the path.
        """
        clone_dir = super().setup_checkout(checkout_dir)

        if clone_dir is None:
            return None

        orig_clone_dir = os.path.join(checkout_dir, 'orig')
        child_clone_dir = os.path.join(checkout_dir, 'child')
        grandchild_clone_dir = os.path.join(checkout_dir, 'grandchild')

        cls._run_git(['clone', cls.git_dir, orig_clone_dir])
        cls._run_git(['clone', orig_clone_dir, child_clone_dir])
        cls._run_git(['clone', child_clone_dir, grandchild_clone_dir])

        cls.orig_clone_dir = os.path.realpath(orig_clone_dir)
        cls.child_clone_dir = os.path.realpath(child_clone_dir)
        cls.grandchild_clone_dir = os.path.realpath(grandchild_clone_dir)

        return orig_clone_dir

    def test_check_dependencies_with_git_found(self) -> None:
        """Testing GitClient.check_dependencies with git found"""
        self.spy_on(check_install, op=kgb.SpyOpMatchAny([
            {
                'args': (['git', '--help'],),
                'op': kgb.SpyOpReturn(True),
            },
            {
                'args': (['git.cmd', '--help'],),
                'op': kgb.SpyOpReturn(True),
            },
        ]))

        client = self.build_client(setup=False)
        client.check_dependencies()

        self.assertSpyCallCount(check_install, 1)
        self.assertSpyCalledWith(check_install, ['git', '--help'])

        self.assertEqual(client.git, 'git')

    def test_check_dependencies_with_gitcmd_found_on_windows(self) -> None:
        """Testing GitClient.check_dependencies with git.cmd found on Windows
        """
        self.spy_on(
            get_git_candidates,
            op=kgb.SpyOpReturn(get_git_candidates(target_platform='windows')))

        self.spy_on(check_install, op=kgb.SpyOpMatchAny([
            {
                'args': (['git', '--help'],),
                'op': kgb.SpyOpReturn(False),
            },
            {
                'args': (['git.cmd', '--help'],),
                'op': kgb.SpyOpReturn(True),
            },
        ]))

        client = self.build_client(setup=False)
        client.check_dependencies()

        self.assertSpyCallCount(check_install, 2)
        self.assertSpyCalledWith(check_install.calls[0], ['git', '--help'])
        self.assertSpyCalledWith(check_install.calls[1], ['git.cmd', '--help'])

        self.assertEqual(client.git, 'git.cmd')

    def test_check_dependencies_with_missing(self) -> None:
        """Testing GitClient.check_dependencies with dependencies
        missing
        """
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        message = "Command line tools ('git') are missing."

        with self.assertRaisesMessage(SCMClientDependencyError, message):
            client.check_dependencies()

        self.assertSpyCallCount(check_install, 1)
        self.assertSpyCalledWith(check_install, ['git', '--help'])

    def test_check_dependencies_with_missing_on_windows(self) -> None:
        """Testing GitClient.check_dependencies with dependencies
        missing on Windows
        """
        self.spy_on(
            get_git_candidates,
            op=kgb.SpyOpReturn(get_git_candidates(target_platform='windows')))

        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        message = "Command line tools (one of ('git', 'git.cmd')) are missing."

        with self.assertRaisesMessage(SCMClientDependencyError, message):
            client.check_dependencies()

        self.assertSpyCallCount(check_install, 2)
        self.assertSpyCalledWith(check_install, ['git', '--help'])
        self.assertSpyCalledWith(check_install, ['git.cmd', '--help'])

    def test_git_with_deps_missing(self) -> None:
        """Testing GitClient.git with dependencies missing"""
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        # Make sure dependencies are checked for this test before we run
        # git(). This will be the expected setup flow.
        self.assertFalse(client.has_dependencies())

        # This will fall back to "git" even if dependencies are missing.
        self.assertEqual(client.git, 'git')

        self.assertSpyCallCount(check_install, 1)
        self.assertSpyCalledWith(check_install, ['git', '--help'])

    def test_git_with_deps_not_checked(self) -> None:
        """Testing GitClient.git with dependencies not
        checked
        """
        # A False value is used just to ensure git() bails early,
        # and to minimize side-effects.
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        message = re.escape(
            'Either GitClient.setup() or '
            'GitClient.has_dependencies() must be called before other '
            'functions are used.'
        )

        with self.assertRaisesRegex(SCMError, message):
            client.git

    def test_get_local_path_with_deps_missing(self) -> None:
        """Testing GitClient.get_local_path with dependencies missing"""
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        # Make sure dependencies are checked for this test before we run
        # get_local_path(). This will be the expected setup flow.
        self.assertFalse(client.has_dependencies())

        with self.assertLogs(level='DEBUG') as ctx:
            local_path = client.get_local_path()

        self.assertIsNone(local_path)

        self.assertEqual(
            ctx.records[0].msg,
            'Unable to execute "git --help" or "git.cmd --help": skipping Git')

        self.assertSpyCallCount(check_install, 1)
        self.assertSpyCalledWith(check_install, ['git', '--help'])

    def test_get_local_path_with_deps_not_checked(self) -> None:
        """Testing GitClient.get_local_path with dependencies not
        checked
        """
        # A False value is used just to ensure get_local_path() bails early,
        # and to minimize side-effects.
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        message = re.escape(
            'Either GitClient.setup() or '
            'GitClient.has_dependencies() must be called before other '
            'functions are used.'
        )

        with self.assertRaisesRegex(SCMError, message):
            client.get_local_path()

    def test_get_repository_info_simple(self) -> None:
        """Testing GitClient get_repository_info, simple case"""
        client = self.build_client()
        ri = client.get_repository_info()
        assert ri is not None
        assert isinstance(ri.path, str)
        assert ri.base_path is not None

        self.assertIsInstance(ri, RepositoryInfo)
        self.assertEqual(ri.base_path, '')
        self.assertEqual(ri.path.rstrip('/.git'), self.git_dir)

    def test_get_repository_info_with_deps_missing(self) -> None:
        """Testing GitClient.get_repository_info with dependencies
        missing
        """
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        # Make sure dependencies are checked for this test before we run
        # get_repository_info(). This will be the expected setup flow.
        self.assertFalse(client.has_dependencies())

        with self.assertLogs(level='DEBUG') as ctx:
            repository_info = client.get_repository_info()

        self.assertIsNone(repository_info)

        self.assertEqual(
            ctx.records[0].msg,
            'Unable to execute "git --help" or "git.cmd --help": skipping Git')

        self.assertSpyCallCount(check_install, 1)
        self.assertSpyCalledWith(check_install, ['git', '--help'])

    def test_get_repository_info_with_deps_not_checked(self) -> None:
        """Testing GitClient.get_repository_info with dependencies
        not checked
        """
        # A False value is used just to ensure get_repository_info() bails
        # early, and to minimize side-effects.
        self.spy_on(check_install, op=kgb.SpyOpReturn(False))

        client = self.build_client(setup=False)

        message = re.escape(
            'Either GitClient.setup() or '
            'GitClient.has_dependencies() must be called before other '
            'functions are used.'
        )

        with self.assertRaisesRegex(SCMError, message):
            client.get_repository_info()

    def test_scan_for_server_simple(self) -> None:
        """Testing GitClient scan_for_server, simple case"""
        client = self.build_client()
        ri = client.get_repository_info()
        assert ri is not None

        server = client.scan_for_server(ri)
        self.assertIsNone(server)

    def test_scan_for_server_property(self) -> None:
        """Testing GitClient scan_for_server using repo property"""
        client = self.build_client()

        self._run_git(['config', 'reviewboard.url', self.TESTSERVER])
        ri = client.get_repository_info()
        assert ri is not None

        self.assertEqual(client.scan_for_server(ri), self.TESTSERVER)

    def test_diff(self) -> None:
        """Testing GitClient.diff"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()
        base_commit_id = self._git_get_head()

        self._git_add_file_commit('foo.txt', FOO1, 'delete and modify stuff')
        commit_id = self._git_get_head()

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'base_commit_id': base_commit_id,
                'commit_id': commit_id,
                'diff': (
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@ multa quoque et bello passus, '
                    b'dum conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_multiple_commits(self) -> None:
        """Testing GitClient.diff with multiple commits"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()

        base_commit_id = self._git_get_head()

        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        self._git_add_file_commit('foo.txt', FOO2, 'commit 1')
        self._git_add_file_commit('foo.txt', FOO3, 'commit 1')
        commit_id = self._git_get_head()

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'base_commit_id': base_commit_id,
                'commit_id': commit_id,
                'diff': (
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'63036ed3fcafe870d567a14dd5884f4fed70126c 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -1,12 +1,11 @@\n ARMA virumque cano, Troiae '
                    b'qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' Italiam, fato profugus, Laviniaque venit\n'
                    b' litora, multum ille et terris iactatus et alto\n'
                    b' vi superum saevae memorem Iunonis ob iram;\n'
                    b'-multa quoque et bello passus, dum conderet urbem,\n'
                    b'+dum conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b'+Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_exclude_patterns(self) -> None:
        """Testing GitClient.diff with file exclusion"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()
        base_commit_id = self._git_get_head()

        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        self._git_add_file_commit('exclude.txt', FOO2, 'commit 2')
        commit_id = self._git_get_head()

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions, exclude_patterns=['exclude.txt']),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@ multa quoque et bello passus, dum '
                    b'conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_exclude_patterns_spaces_in_filename(self) -> None:
        """Testing GitClient.diff with file exclusion and spaces in filename"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()
        base_commit_id = self._git_get_head()

        self._git_add_file_commit('included file.txt', FOO1, 'commit 1')
        self._git_add_file_commit('excluded file.txt', FOO2, 'commit 2')
        commit_id = self._git_get_head()

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions, include_files=[],
                        exclude_patterns=['excluded file.txt']),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'diff --git a/included file.txt b/included file.txt\n'
                    b'new file mode 100644\nindex '
                    b'0000000000000000000000000000000000000000..'
                    b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb\n'
                    b'--- /dev/null\n'
                    b'+++ b/included file.txt\t\n'
                    b'@@ -0,0 +1,9 @@\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+Italiam, fato profugus, Laviniaque venit\n'
                    b'+litora, multum ille et terris iactatus et alto\n'
                    b'+vi superum saevae memorem Iunonis ob iram;\n'
                    b'+multa quoque et bello passus, dum conderet urbem,\n'
                    b'+inferretque deos Latio, genus unde Latinum,\n'
                    b'+Albanique patres, atque altae moenia Romae.\n'
                    b'+Musa, mihi causas memora, quo numine laeso,\n'
                    b'+\n'
                ),
                'parent_diff': None,
            })

    def test_diff_exclude_in_subdir(self) -> None:
        """Testing GitClient.diff with file exclusion in a subdir"""
        client = self.build_client(needs_diff=True)
        base_commit_id = self._git_get_head()

        os.mkdir('subdir')
        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        self._git_add_file_commit('subdir/exclude.txt', FOO2, 'commit 2')

        os.chdir('subdir')
        client.get_repository_info()

        commit_id = self._git_get_head()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions, exclude_patterns=['exclude.txt']),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@ multa quoque et bello passus, dum '
                    b'conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_exclude_patterns_root_pattern_in_subdir(self) -> None:
        """Testing GitClient.diff with file exclusion in the repo root"""
        client = self.build_client(needs_diff=True)
        base_commit_id = self._git_get_head()

        os.mkdir('subdir')
        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        self._git_add_file_commit('exclude.txt', FOO2, 'commit 2')
        os.chdir('subdir')

        client.get_repository_info()

        commit_id = self._git_get_head()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions,
                        exclude_patterns=[os.path.sep + 'exclude.txt']),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@ multa quoque et bello passus, dum '
                    b'conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_exclude_patterns_and_moved_file(self) -> None:
        """Testing GitClient.diff with file exclusion and moved file"""
        client = self.build_client(
            needs_diff=True,
            caps={
                'diffs': {
                    'moved_files': True,
                },
            })
        client.get_repository_info()

        self._git_add_file_commit('original.txt', FOO1, 'create original.txt')

        base_commit_id = self._git_get_head()

        self._run_git(['mv', 'original.txt', 'renamed.txt'])
        self._run_git(['commit', '-m', 'rename original.txt to renamed.txt'])

        self._git_add_file_commit('exclude.txt', FOO2, 'add exclude.txt')

        commit_id = self._git_get_head()
        revisions = client.parse_revision_spec([base_commit_id, commit_id])

        result = client.diff(revisions, exclude_patterns=['exclude.txt'])

        diff_content = result['diff']
        assert diff_content is not None

        self.assertIn(b'renamed.txt', diff_content)
        self.assertNotIn(b'exclude.txt', diff_content)

    def test_diff_with_branch_diverge(self) -> None:
        """Testing GitClient.diff with divergent branches"""
        client = self.build_client(needs_diff=True)

        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        self._run_git(['checkout', '-b', 'mybranch', '--track',
                      'origin/master'])
        base_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO2, 'commit 2')
        commit_id = self._git_get_head()
        client.get_repository_info()

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'e619c1387f5feb91f0ca83194650bfe4f6c2e347 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -1,4 +1,6 @@\n'
                    b' ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' Italiam, fato profugus, Laviniaque venit\n'
                    b' litora, multum ille et terris iactatus et alto\n'
                    b' vi superum saevae memorem Iunonis ob iram;\n'
                    b'@@ -6,7 +8,4 @@ multa quoque et bello passus, dum '
                    b'conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

        self._run_git(['checkout', 'master'])
        client.get_repository_info()
        commit_id = self._git_get_head()

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@ multa quoque et bello passus, dum '
                    b'conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_tracking_branch_no_origin(self) -> None:
        """Testing GitClient.diff with a tracking branch, but no origin remote
        """
        client = self.build_client(needs_diff=True)

        self._run_git(['remote', 'add', 'quux', self.git_dir])
        self._run_git(['fetch', 'quux'])
        self._run_git(['checkout', '-b', 'mybranch', '--track', 'quux/master'])

        base_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO1, 'delete and modify stuff')
        commit_id = self._git_get_head()

        client.get_repository_info()

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@ multa quoque et bello passus, dum '
                    b'conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_tracking_branch_local(self) -> None:
        """Testing GitClient.diff with a local tracking branch"""
        client = self.build_client(needs_diff=True)

        base_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')

        self._run_git(['checkout', '-b', 'mybranch', '--track', 'master'])
        self._git_add_file_commit('foo.txt', FOO2, 'commit 2')
        commit_id = self._git_get_head()

        client.get_repository_info()

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'e619c1387f5feb91f0ca83194650bfe4f6c2e347 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -1,4 +1,6 @@\n'
                    b' ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' Italiam, fato profugus, Laviniaque venit\n'
                    b' litora, multum ille et terris iactatus et alto\n'
                    b' vi superum saevae memorem Iunonis ob iram;\n'
                    b'@@ -6,7 +8,4 @@ multa quoque et bello passus, dum '
                    b'conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_tracking_branch_option(self) -> None:
        """Testing GitClient.diff with option override for tracking branch"""
        client = self.build_client(
            needs_diff=True,
            options={
                'tracking': 'origin/master',
            })

        self._run_git(['remote', 'add', 'bad', self.git_dir])
        self._run_git(['fetch', 'bad'])
        self._run_git(['checkout', '-b', 'mybranch', '--track', 'bad/master'])

        base_commit_id = self._git_get_head()

        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        commit_id = self._git_get_head()

        client.get_repository_info()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,7 +6,4 @@ multa quoque et bello passus, dum '
                    b'conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_tracking_branch_slash(self) -> None:
        """Testing GitClient.diff with tracking branch that has slash in its
        name
        """
        client = self.build_client(needs_diff=True)

        self._run_git(['fetch', 'origin'])
        self._run_git(['checkout', '-b', 'my/branch', '--track',
                       'origin/not-master'])
        base_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO2, 'commit 2')
        commit_id = self._git_get_head()

        client.get_repository_info()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 5e98e9540e1b741b5be24fcb33c40c1c8069c1fb..'
                    b'e619c1387f5feb91f0ca83194650bfe4f6c2e347 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -1,4 +1,6 @@\n'
                    b' ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' Italiam, fato profugus, Laviniaque venit\n'
                    b' litora, multum ille et terris iactatus et alto\n'
                    b' vi superum saevae memorem Iunonis ob iram;\n'
                ),
                'parent_diff': None,
            })

    def test_parse_revision_spec_no_args(self) -> None:
        """Testing GitClient.parse_revision_spec with no specified revisions"""
        client = self.build_client()

        base_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO2, 'Commit 2')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        self.assertEqual(
            client.parse_revision_spec([]),
            {
                'base': base_commit_id,
                'commit_id': tip_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_no_args_parent(self) -> None:
        """Testing GitClient.parse_revision_spec with no specified revisions
        and a parent diff
        """
        client = self.build_client(options={
            'parent_branch': 'parent-branch',
        })
        parent_base_commit_id = self._git_get_head()

        self._run_git(['fetch', 'origin'])
        self._run_git(['checkout', '-b', 'parent-branch', '--track',
                       'origin/not-master'])
        parent_base_commit_id = self._git_get_head()

        self._git_add_file_commit('foo.txt', FOO2, 'Commit 2')
        base_commit_id = self._git_get_head()

        self._run_git(['checkout', '-b', 'topic-branch'])

        self._git_add_file_commit('foo.txt', FOO3, 'Commit 3')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        self.assertEqual(
            client.parse_revision_spec([]),
            {
                'base': base_commit_id,
                'commit_id': tip_commit_id,
                'parent_base': parent_base_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_one_arg(self) -> None:
        """Testing GitClient.parse_revision_spec with one specified revision"""
        client = self.build_client()

        base_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO2, 'Commit 2')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        self.assertEqual(
            client.parse_revision_spec([tip_commit_id]),
            {
                'base': base_commit_id,
                'commit_id': tip_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_one_arg_parent(self) -> None:
        """Testing GitClient.parse_revision_spec with one specified revision
        and a parent diff
        """
        client = self.build_client()

        parent_base_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO2, 'Commit 2')
        base_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO3, 'Commit 3')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        self.assertEqual(
            client.parse_revision_spec([tip_commit_id]),
            {
                'base': base_commit_id,
                'commit_id': tip_commit_id,
                'parent_base': parent_base_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_two_args(self) -> None:
        """Testing GitClient.parse_revision_spec with two specified
        revisions
        """
        client = self.build_client()

        base_commit_id = self._git_get_head()
        self._run_git(['checkout', '-b', 'topic-branch'])
        self._git_add_file_commit('foo.txt', FOO2, 'Commit 2')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        self.assertEqual(
            client.parse_revision_spec(['master', 'topic-branch']),
            {
                'base': base_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_one_arg_two_revs(self) -> None:
        """Testing GitClient.parse_revision_spec with diff-since syntax"""
        client = self.build_client()

        base_commit_id = self._git_get_head()
        self._run_git(['checkout', '-b', 'topic-branch'])
        self._git_add_file_commit('foo.txt', FOO2, 'Commit 2')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        self.assertEqual(
            client.parse_revision_spec(['master...topic-branch']),
            {
                'base': base_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_one_arg_since_merge(self) -> None:
        """Testing GitClient.parse_revision_spec with diff-since-merge
        syntax
        """
        client = self.build_client()

        base_commit_id = self._git_get_head()
        self._run_git(['checkout', '-b', 'topic-branch'])
        self._git_add_file_commit('foo.txt', FOO2, 'Commit 2')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        self.assertEqual(
            client.parse_revision_spec(['master...topic-branch']),
            {
                'base': base_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_with_too_many_revisions(self) -> None:
        """Testing GitClient.parse_revision_spec with too many revisions"""
        client = self.build_client()

        with self.assertRaises(TooManyRevisionsError):
            client.parse_revision_spec(['1', '2', '3'])

    def test_parse_revision_spec_with_diff_finding_parent(self) -> None:
        """Testing GitClient.parse_revision_spec with target branch off a
        tracking branch not aligned with the remote
        """
        client = self.build_client(needs_diff=True)

        # In this case, the parent must be the non-aligned tracking branch
        # and the parent_base must be the remote tracking branch.
        client.get_repository_info()

        self._git_add_file_commit('foo.txt', FOO1, 'on master')
        self._run_git(['checkout', 'not-master'])  # A remote branch
        parent_base_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO2, 'on not-master')
        parent_commit_id = self._git_get_head()
        self._run_git(['checkout', '-b', 'topic-branch'])
        self._git_add_file_commit('foo.txt', FOO3, 'commit 2')
        self._git_add_file_commit('foo.txt', FOO4, 'commit 3')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        # revisions =
        #     {
        #         'base': u'357c1b9',
        #         'tip': u'10c5cd3',
        #         'parent_base': u'0e88e51',
        #     }
        #
        # `git log --graph --all --decorate --oneline` =
        #     * 7c17015 (master) on master
        #     | * 10c5cd3 (HEAD -> topic-branch) commit 3
        #     | * 00c99f9 commit 2
        #     | * 357c1b9 (not-master) on not-master
        #     | * 0e88e51 (origin/not-master) Commit 2
        #     |/
        #     * 18c5c09 (origin/master, origin/HEAD) Commit 1
        #     * e6a3577 Initial Commit
        self.assertEqual(
            client.parse_revision_spec(['topic-branch', '^not-master']),
            {
                'base': parent_commit_id,
                'parent_base': parent_base_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_with_diff_finding_parent_case_one(
        self,
    ) -> None:
        """Testing GitClient.parse_revision_spec with target branch off a
        tracking branch aligned with the remote
        """
        client = self.build_client(
            needs_diff=True,
            options={
                'tracking': 'origin/not-master',
            })

        # In this case, the parent_base should be the tracking branch aligned
        # with the remote.
        client.get_repository_info()

        self._run_git(['fetch', 'origin'])
        self._run_git(['checkout', '-b', 'not-master',
                       '--track', 'origin/not-master'])
        parent_commit_id = self._git_get_head()
        self._run_git(['checkout', '-b', 'feature-branch'])
        self._git_add_file_commit('foo.txt', FOO3, 'on feature-branch')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        # revisions =
        #     {
        #         'commit_id': u'0a5734a',
        #         'base': u'0e88e51',
        #         'tip': u'0a5734a',
        #     }
        #
        # `git log --graph --all --decorate --oneline` =
        #     * 0a5734a (HEAD -> feature-branch) on feature-branch
        #     * 0e88e51 (origin/not-master, not-master) Commit 2
        #     * 18c5c09 (origin/master, origin/HEAD, master) Commit 1
        #     * e6a3577 Initial Commit
        #
        # Because parent_base == base, parent_base will not be in revisions.
        self.assertEqual(
            client.parse_revision_spec([]),
            {
                'base': parent_commit_id,
                'commit_id': tip_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_with_diff_finding_parent_case_two(
        self,
    ) -> None:
        """Testing GitClient.parse_revision_spec with target branch off
        a tracking branch with changes since the remote
        """
        client = self.build_client(needs_diff=True)

        # In this case, the parent_base must be the remote tracking branch,
        # despite the fact that it is a few changes behind.
        client.get_repository_info()

        self._run_git(['fetch', 'origin'])
        self._run_git(['checkout', '-b', 'not-master',
                       '--track', 'origin/not-master'])
        parent_base_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO2, 'on not-master')
        parent_commit_id = self._git_get_head()
        self._run_git(['checkout', '-b', 'feature-branch'])
        self._git_add_file_commit('foo.txt', FOO3, 'on feature-branch')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        # revisions =
        #     {
        #         'base': u'b0f5d74',
        #         'tip': u'8b5d1b9',
        #         'parent_base': u'0e88e51',
        #     }
        #
        # `git log --graph --all --decorate --oneline` =
        #     * 8b5d1b9 (HEAD -> feature-branch) on feature-branch
        #     * b0f5d74 (not-master) on not-master
        #     * 0e88e51 (origin/not-master) Commit 2
        #     * 18c5c09 (origin/master, origin/HEAD, master) Commit 1
        #     * e6a3577 Initial Commit
        self.assertEqual(
            client.parse_revision_spec(['feature-branch', '^not-master']),
            {
                'base': parent_commit_id,
                'parent_base': parent_base_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_with_diff_finding_parent_case_three(
        self,
    ) -> None:
        """Testing GitClient.parse_revision_spec with target branch off a
        branch not properly tracking the remote
        """
        client = self.build_client(needs_diff=True)

        # In this case, the parent_base must be the remote tracking branch,
        # even though it is not properly being tracked.
        client.get_repository_info()

        self._run_git(['branch', '--no-track', 'not-master',
                       'origin/not-master'])
        self._run_git(['checkout', 'not-master'])
        parent_commit_id = self._git_get_head()
        self._run_git(['checkout', '-b', 'feature-branch'])
        self._git_add_file_commit('foo.txt', FOO3, 'on feature-branch')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        # revisions =
        #     {
        #         'base': u'0e88e51',
        #         'tip': u'58981f2',
        #     }
        #
        # `git log --graph --all --decorate --oneline` =
        #     * 58981f2 (HEAD -> feature-branch) on feature-branch
        #     * 0e88e51 (origin/not-master, not-master) Commit 2
        #     * 18c5c09 (origin/master, origin/HEAD, master) Commit 1
        #     * e6a3577 Initial Commit
        self.assertEqual(
            client.parse_revision_spec(['feature-branch', '^not-master']),
            {
                'base': parent_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_with__diff_finding_parent_case_four(
        self,
    ) -> None:
        """Testing GitClient.parse_revision_spec with a target branch that
        merged a tracking branch off another tracking branch
        """
        client = self.build_client(needs_diff=True)

        # In this case, the parent_base must be the base of the merge, because
        # the user will expect that the diff would show the merged changes.
        client.get_repository_info()

        self._run_git(['checkout', 'master'])
        parent_commit_id = self._git_get_head()
        self._run_git(['checkout', '-b', 'feature-branch'])
        self._git_add_file_commit('foo.txt', FOO1, 'on feature-branch')
        self._run_git(['merge', 'origin/not-master'])
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        # revisions =
        #     {
        #         'commit_id': u'bef8dcd',
        #         'base': u'18c5c09',
        #         'tip': u'bef8dcd',
        #     }
        #
        # `git log --graph --all --decorate --oneline` =
        #     *   bef8dcd (HEAD -> feature-branch) Merge remote-tracking branch
        #                 'origin/not-master' into feature-branch
        #     |\
        #     | * 0e88e51 (origin/not-master) Commit 2
        #     * | a385539 on feature-branch
        #     |/
        #     * 18c5c09 (origin/master, origin/HEAD, master) Commit 1
        #     * e6a3577 Initial Commit
        self.assertEqual(
            client.parse_revision_spec([]),
            {
                'base': parent_commit_id,
                'commit_id': tip_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_with_diff_finding_parent_case_five(
        self,
    ) -> None:
        """Testing GitClient.parse_revision_spec with a target branch posted
        off a tracking branch that merged another tracking branch
        """
        client = self.build_client(
            needs_diff=True,
            options={
                'tracking': 'origin/not-master',
            })

        # In this case, the parent_base must be tracking branch that merged
        # the other tracking branch.
        client.get_repository_info()

        self._git_add_file_commit('foo.txt', FOO2, 'on master')
        self._run_git(['checkout', '-b', 'not-master',
                       '--track', 'origin/not-master'])
        self._run_git(['merge', 'origin/master'])
        parent_commit_id = self._git_get_head()
        self._run_git(['checkout', '-b', 'feature-branch'])
        self._git_add_file_commit('foo.txt', FOO4, 'on feature-branch')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        # revisions =
        #     {
        #         'commit_id': u'ebf2e89',
        #         'base': u'0e88e51',
        #         'tip': u'ebf2e89'
        #     }
        #
        # `git log --graph --all --decorate --oneline` =
        #     * ebf2e89 (HEAD -> feature-branch) on feature-branch
        #     * 0e88e51 (origin/not-master, not-master) Commit 2
        #     | * 7e202ff (master) on master
        #     |/
        #     * 18c5c09 (origin/master, origin/HEAD) Commit 1
        #     * e6a3577 Initial Commit
        self.assertEqual(
            client.parse_revision_spec([]),
            {
                'base': parent_commit_id,
                'commit_id': tip_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_with_diff_finding_parent_case_six(
        self,
    ) -> None:
        """Testing GitClient.parse_revision_spec with a target branch posted
        off a remote branch without any tracking branches
        """
        client = self.build_client(needs_diff=True)

        # In this case, the parent_base must be remote tracking branch. The
        # existence of a tracking branch shouldn't matter much.
        client.get_repository_info()

        self._run_git(['checkout', '-b', 'feature-branch',
                       'origin/not-master'])
        parent_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO2, 'on feature-branch')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        # revisions =
        #     {
        #         'commit_id': u'19da590',
        #         'base': u'0e88e51',
        #         'tip': u'19da590',
        #     }
        #
        # `git log --graph --all --decorate --oneline` =
        #     * 19da590 (HEAD -> feature-branch) on feature-branch
        #     * 0e88e51 (origin/not-master) Commit 2
        #     * 18c5c09 (origin/master, origin/HEAD, master) Commit 1
        #     * e6a3577 Initial Commit
        self.assertEqual(
            client.parse_revision_spec([]),
            {
                'base': parent_commit_id,
                'commit_id': tip_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_with_diff_finding_parent_case_seven(
        self,
    ) -> None:
        """Testing GitClient.parse_revision_spec with a target branch posted
        off a remote branch that is aligned to the same commit as another
        remote branch
        """
        client = self.build_client(needs_diff=True)

        # In this case, the parent_base must be common commit that the two
        # remote branches are aligned to.
        client.get_repository_info()

        # Since pushing data upstream to the test repo corrupts its state,
        # we need to use the child clone.
        os.chdir(self.child_clone_dir)

        client.get_repository_info()

        self._run_git(['checkout', '-b', 'remote-branch1'])
        self._git_add_file_commit('foo1.txt', FOO1, 'on remote-branch1')
        self._run_git(['push', 'origin', 'remote-branch1'])
        self._run_git(['checkout', '-b', 'remote-branch2'])
        self._git_add_file_commit('foo2.txt', FOO1, 'on remote-branch2')
        self._run_git(['push', 'origin', 'remote-branch2'])

        self._run_git(['checkout', 'master'])
        self._run_git(['merge', 'remote-branch1'])
        self._run_git(['merge', 'remote-branch2'])
        self._git_add_file_commit('foo3.txt', FOO1, 'on master')
        parent_commit_id = self._git_get_head()

        self._run_git(['push', 'origin', 'master:remote-branch1'])
        self._run_git(['push', 'origin', 'master:remote-branch2'])

        self._run_git(['checkout', '-b', 'feature-branch'])
        self._git_add_file_commit('foo4.txt', FOO1, 'on feature-branch')

        tip_commit_id = self._git_get_head()

        # revisions =
        #     {
        #         'base': u'bf0036b',
        #         'tip': u'dadae87',
        #     }
        #
        # `git log --graph --all --decorate --oneline` =
        #     * dadae87 (HEAD -> feature-branch) on feature-branch
        #     * bf0036b (origin/remote-branch2, origin/remote-branch1, master)
        #                                                            on master
        #     * 5f48441 (remote-branch2) on remote-branch2
        #     * eb40eaf (remote-branch1) on remote-branch1
        #     * 18c5c09 (origin/master, origin/HEAD) Commit 1
        #     * e6a3577 Initial Commit
        self.assertEqual(
            client.parse_revision_spec(['feature-branch', '^master']),
            {
                'base': parent_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_with_diff_finding_parent_case_eight(
        self,
    ) -> None:
        """Testing GitClient.parse_revision_spec with a target branch not
        up-to-date with a remote branch
        """
        client = self.build_client(needs_diff=True)

        # In this case, there is no good way of detecting the remote branch we
        # are not up-to-date with, so the parent_base must be the common commit
        # that the target branch and remote branch share.
        client.get_repository_info()

        # Since pushing data upstream to the test repo corrupts its state,
        # we need to use the child clone.
        os.chdir(self.child_clone_dir)

        client.get_repository_info()

        self._run_git(['checkout', 'master'])
        self._git_add_file_commit('foo.txt', FOO1, 'on master')

        parent_base_commit_id = self._git_get_head()

        self._run_git(['checkout', '-b', 'remote-branch1'])
        self._git_add_file_commit('foo1.txt', FOO1, 'on remote-branch1')
        self._run_git(['push', 'origin', 'remote-branch1'])

        self._run_git(['checkout', 'master'])
        self._git_add_file_commit('foo2.txt', FOO1, 'on master')
        parent_commit_id = self._git_get_head()

        self._run_git(['checkout', '-b', 'feature-branch'])
        self._git_add_file_commit('foo3.txt', FOO1, 'on feature-branch')

        client.get_repository_info()
        tip_commit_id = self._git_get_head()

        # revisions =
        #     {
        #         'base': u'318f050',
        #         'tip': u'6e37a00',
        #         'parent_base': u'0ff6635'
        #     }
        #
        # `git log --graph --all --decorate --oneline` =
        #     * 6e37a00 (HEAD -> feature-branch) on feature-branch
        #     * 318f050 (master) on master
        #     | * 9ad7b1f (origin/remote-branch1, remote-branch1)
        #     |/                                on remote-branch1
        #     * 0ff6635 on master
        #     * 18c5c09 (origin/master, origin/HEAD) Commit 1
        #     * e6a3577 Initial Commit
        self.assertEqual(
            client.parse_revision_spec(['feature-branch', '^master']),
            {
                'base': parent_commit_id,
                'parent_base': parent_base_commit_id,
                'tip': tip_commit_id,
            })

    def test_parse_revision_spec_with_diff_finding_parent_case_nine(
        self,
    ) -> None:
        """Testing GitClient.parse_revision_spec with a target branch that has
        branches from different remotes in its path
        """
        client = self.build_client(needs_diff=True)

        # In this case, the other remotes should be ignored and the parent_base
        # should be some origin/*.
        client.get_repository_info()
        self._run_git(['checkout', 'not-master'])

        # Since pushing data upstream to the test repo corrupts its state,
        # we need to use the child clone.
        os.chdir(self.grandchild_clone_dir)

        client.get_repository_info()

        # Adding the original clone as a second remote to our repository.
        self._run_git(['remote', 'add', 'not-origin', self.orig_clone_dir])
        self._run_git(['fetch', 'not-origin'])
        parent_base_commit_id = self._git_get_head()

        self._run_git(['checkout', 'master'])
        self._run_git(['merge', 'not-origin/master'])

        self._git_add_file_commit('foo1.txt', FOO1, 'on master')
        self._run_git(['push', 'not-origin', 'master:master'])
        self._git_add_file_commit('foo2.txt', FOO1, 'on master')
        parent_commit_id = self._git_get_head()

        self._run_git(['checkout', '-b', 'feature-branch'])
        self._git_add_file_commit('foo3.txt', FOO1, 'on feature-branch')
        tip_commit_id = self._git_get_head()

        # revisions =
        #     {
        #         'base': u'6f23ed0',
        #         'tip': u'8703f95',
        #         'parent_base': u'18c5c09',
        #     }
        #
        # `git log --graph --all --decorate --oneline` =
        #     * 8703f95 (HEAD -> feature-branch) on feature-branch
        #     * 6f23ed0 (master) on master
        #     * f6236bf (not-origin/master) on master
        #     | * 0e88e51 (origin/not-master, not-origin/not-master) Commit 2
        #     |/
        #     * 18c5c09 (origin/master, origin/HEAD) Commit 1
        #     * e6a3577 Initial Commit
        self.assertEqual(
            client.parse_revision_spec(['feature-branch', '^master']),
            {
                'base': parent_commit_id,
                'parent_base': parent_base_commit_id,
                'tip': tip_commit_id,
            })

    def test_get_raw_commit_message(self) -> None:
        """Testing GitClient.get_raw_commit_message"""
        client = self.build_client()

        self._git_add_file_commit('foo.txt', FOO2, 'Commit 2')
        client.get_repository_info()
        revisions = client.parse_revision_spec([])

        self.assertEqual(client.get_raw_commit_message(revisions),
                         'Commit 2')

    def test_push_upstream_pull_exception(self) -> None:
        """Testing GitClient.push_upstream with an invalid remote branch"""
        client = self.build_client()

        # It must raise a PushError exception because the 'git pull' from an
        # invalid upstream branch will fail.
        with self.assertRaisesMessage(PushError,
                                      'Could not determine remote for branch '
                                      '"non-existent-branch".'):
            client.push_upstream('non-existent-branch')

    def test_push_upstream_no_push_exception(self) -> None:
        """Testing GitClient.push_upstream with 'git push' disabled"""
        client = self.build_client()

        # Set the push url to be an invalid one.
        self._run_git(['remote', 'set-url', '--push', 'origin', 'bad-url'])

        with self.assertRaisesMessage(PushError,
                                      'Could not push branch "master" to '
                                      'upstream.'):
            client.push_upstream('master')

    def test_merge_invalid_destination(self) -> None:
        """Testing GitClient.merge with an invalid destination branch"""
        client = self.build_client()

        # It must raise a MergeError exception because 'git checkout' to the
        # invalid destination branch will fail.
        try:
            client.merge(target='master',
                         destination='non-existent-branch',
                         message='commit message',
                         author=self.AUTHOR)
        except MergeError as e:
            self.assertTrue(str(e).startswith(
                'Could not checkout to branch "non-existent-branch"'))
        else:
            self.fail('Expected MergeError')

    def test_merge_invalid_target(self) -> None:
        """Testing GitClient.merge with an invalid target branch"""
        client = self.build_client()

        # It must raise a MergeError exception because 'git merge' from an
        # invalid target branch will fail.
        try:
            client.merge(target='non-existent-branch',
                         destination='master',
                         message='commit message',
                         author=self.AUTHOR)
        except MergeError as e:
            self.assertTrue(str(e).startswith(
                'Could not merge branch "non-existent-branch"'))
        else:
            self.fail('Expected MergeError')

    def test_merge_with_squash(self) -> None:
        """Testing GitClient.merge with squash set to True"""
        client = self.build_client()
        client.get_repository_info()

        self.spy_on(run_process_exec)

        # Since pushing data upstream to the test repo corrupts its state,
        # we need to use the child clone.
        os.chdir(self.child_clone_dir)

        client.get_repository_info()

        self._run_git(['checkout', '-b', 'new-branch'])
        self._git_add_file_commit('foo1.txt', FOO1, 'on new-branch')
        self._run_git(['push', 'origin', 'new-branch'])

        client.merge(target='new-branch',
                     destination='master',
                     message='message',
                     author=self.AUTHOR,
                     squash=True)

        self.assertSpyCalledWith(
            run_process_exec.calls[-2],
            ['git', 'merge', 'new-branch', '--squash', '--no-commit'])

    def test_merge_without_squash(self) -> None:
        """Testing GitClient.merge with squash set to False"""
        client = self.build_client()
        client.get_repository_info()

        self.spy_on(run_process_exec)

        # Since pushing data upstream to the test repo corrupts its state,
        # we need to use the child clone.
        os.chdir(self.child_clone_dir)

        client.get_repository_info()

        self._run_git(['checkout', '-b', 'new-branch'])
        self._git_add_file_commit('foo1.txt', FOO1, 'on new-branch')
        self._run_git(['push', 'origin', 'new-branch'])

        client.merge(target='new-branch',
                     destination='master',
                     message='message',
                     author=self.AUTHOR,
                     squash=False)

        self.assertSpyCalledWith(
            run_process_exec.calls[-2],
            ['git', 'merge', 'new-branch', '--no-ff', '--no-commit'])

    def test_create_commit_with_run_editor_true(self) -> None:
        """Testing GitClient.create_commit with run_editor set to True"""
        client = self.build_client()

        self.spy_on(run_process_exec)

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        client.create_commit(message='Test commit message.',
                             author=self.AUTHOR,
                             run_editor=True,
                             files=['foo.txt'])

        self.assertSpyLastCalledWith(
            run_process_exec,
            ['git', 'commit', '-m', 'TEST COMMIT MESSAGE.',
             '--author', 'name <email>'])

    def test_create_commit_with_run_editor_false(self) -> None:
        """Testing GitClient.create_commit with run_editor set to False"""
        client = self.build_client()

        self.spy_on(run_process_exec)

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        client.create_commit(message='Test commit message.',
                             author=self.AUTHOR,
                             run_editor=False,
                             files=['foo.txt'])

        self.assertSpyLastCalledWith(
            run_process_exec,
            ['git', 'commit', '-m', 'Test commit message.',
             '--author', 'name <email>'])

    def test_create_commit_with_all_files_true(self) -> None:
        """Testing GitClient.create_commit with all_files set to True"""
        client = self.build_client()

        self.spy_on(run_process_exec)

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        client.create_commit(message='message',
                             author=self.AUTHOR,
                             run_editor=False,
                             files=[],
                             all_files=True)

        self.assertSpyCalledWith(
            run_process_exec.calls[0],
            ['git', 'add', '--all', ':/'])
        self.assertSpyLastCalledWith(
            run_process_exec,
            ['git', 'commit', '-m', 'message', '--author', 'name <email>'])

    def test_create_commit_with_all_files_false(self) -> None:
        """Testing GitClient.create_commit with all_files set to False"""
        client = self.build_client()

        self.spy_on(run_process_exec)

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        client.create_commit(message='message',
                             author=self.AUTHOR,
                             run_editor=False,
                             files=['foo.txt'],
                             all_files=False)

        self.assertSpyCalledWith(
            run_process_exec.calls[0],
            ['git', 'add', 'foo.txt'])
        self.assertSpyLastCalledWith(
            run_process_exec,
            ['git', 'commit', '-m', 'message', '--author', 'name <email>'])

    def test_create_commit_with_empty_commit_message(self) -> None:
        """Testing GitClient.create_commit with empty commit message"""
        client = self.build_client()

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        message = (
            "A commit message wasn't provided. The patched files are in "
            "your tree and are staged for commit, but haven't been "
            "committed. Run `git commit` to commit them."
        )

        with self.assertRaisesMessage(CreateCommitError, message):
            client.create_commit(message='',
                                 author=self.AUTHOR,
                                 run_editor=True,
                                 files=['foo.txt'])

    def test_create_commit_without_author(self) -> None:
        """Testing GitClient.create_commit without author information"""
        client = self.build_client()

        self.spy_on(run_process_exec)

        with open('foo.txt', 'w') as fp:
            fp.write('change')

        client.create_commit(message='Test commit message.',
                             author=None,
                             run_editor=True,
                             files=['foo.txt'])

        self.assertSpyLastCalledWith(
            run_process_exec,
            ['git', 'commit', '-m', 'TEST COMMIT MESSAGE.'])

    def test_delete_branch_with_merged_only(self) -> None:
        """Testing GitClient.delete_branch with merged_only set to True"""
        client = self.build_client()

        self.spy_on(run_process_exec)

        self._run_git(['branch', 'new-branch'])

        client.delete_branch('new-branch', merged_only=True)

        self.assertSpyLastCalledWith(
            run_process_exec,
            ['git', 'branch', '-d', 'new-branch'])

    def test_delete_branch_without_merged_only(self) -> None:
        """Testing GitClient.delete_branch with merged_only set to False"""
        client = self.build_client()

        self.spy_on(run_process_exec)

        self._run_git(['branch', 'new-branch'])

        client.delete_branch('new-branch', merged_only=False)

        self.assertSpyLastCalledWith(
            run_process_exec,
            ['git', 'branch', '-D', 'new-branch'])

    def test_get_parent_branch_with_non_master_default(self) -> None:
        """Testing GitClient._get_parent_branch with a non-master default
        branch
        """
        client = self.build_client()

        # Since pushing data upstream to the test repo corrupts its state,
        # we need to use the child clone.
        os.chdir(self.child_clone_dir)

        self._run_git(['branch', '-m', 'master', 'main'])
        self._run_git(['push', '-u', 'origin', 'main'])

        client.get_repository_info()

        self.assertEqual(client._get_parent_branch(), 'origin/main')

    def test_get_file_content(self) -> None:
        """Testing GitClient.get_file_content"""
        client = self.build_client()

        self._git_add_file_commit('foo.txt', FOO1, 'delete and modify stuff')

        content = client.get_file_content(
            filename='foo.txt',
            revision='5e98e9540e1b741b5be24fcb33c40c1c8069c1fb')

        self.assertEqual(content, FOO1)

    def test_get_file_content_invalid_revision(self) -> None:
        """Testing GitClient.get_file_content with an invalid revision"""
        client = self.build_client()

        self._git_add_file_commit('foo.txt', FOO1, 'delete and modify stuff')

        with self.assertRaises(SCMError):
            client.get_file_content(
                filename='foo.txt',
                revision='5e98e9540e1b741b5be240000000000000000000')

    def test_get_file_size(self) -> None:
        """Testing GitClient.get_file_size"""
        client = self.build_client()

        self._git_add_file_commit('foo.txt', FOO1, 'delete and modify stuff')

        size = client.get_file_size(
            filename='foo.txt',
            revision='5e98e9540e1b741b5be24fcb33c40c1c8069c1fb')

        self.assertEqual(size, len(FOO1))

    def test_get_file_size_invalid_revision(self) -> None:
        """Testing GitClient.get_file_size with an invalid revision"""
        client = self.build_client()

        self._git_add_file_commit('foo.txt', FOO1, 'delete and modify stuff')

        with self.assertRaises(SCMError):
            client.get_file_size(
                filename='foo.txt',
                revision='5e98e9540e1b741b5be240000000000000000000')


class GitPerforceClientTests(BaseGitClientTests):
    """Unit tests for GitClient wrapping Perforce.

    Version Added:
        4.0
    """

    @classmethod
    def setUpClass(cls) -> None:
        """Set up the test case class."""
        if not is_exe_in_path('p4'):
            raise unittest.SkipTest('p4 executable not available')

        super().setUpClass()

    @classmethod
    def setup_checkout(
        cls,
        checkout_dir: str,
    ) -> str | None:
        """Populate a Git-P4 checkout.

        This will create a fake Perforce upstream with commits containing
        git-p4 change information and depot paths, along with a clone for
        tests.

        Args:
            checkout_dir (str):
                The top-level directory in which the clones will be placed.

        Returns:
            The main clone directory, or ``None`` if :command:`git` isn't in
            the path.
        """
        clone_dir = super().setup_checkout(checkout_dir)

        if clone_dir is None:
            return None

        p4_origin_clone_dir = os.path.join(clone_dir, 'p4-origin')
        p4_clone_dir = os.path.join(clone_dir, 'p4-clone')

        # Create the p4 "remote".
        cls._run_git(['clone', cls.git_dir, p4_origin_clone_dir])
        os.chdir(p4_origin_clone_dir)

        cls._git_add_file_commit(
            filename='existing-file.txt',
            data=FOO2,
            msg=(
                'Add a file to the base clone.\n'
                '\n'
                '[git-p4: depot-paths = "//depot/": change = 5]\n'
            ))

        # Create the clone for the tests.
        cls._run_git(['clone', '-o', 'p4', p4_origin_clone_dir, p4_clone_dir])
        os.chdir(p4_clone_dir)
        cls._run_git(['fetch', 'p4'])
        cls._run_git(['config', '--local', '--add', 'git-p4.port',
                      'example.com:1666'])

        return os.path.realpath(p4_clone_dir)

    def test_get_repository_info(self) -> None:
        """Testing GitClient.get_repository_info with git-p4"""
        client = self.build_client()
        repository_info = client.get_repository_info()
        assert repository_info is not None

        self.assertEqual(repository_info.path, 'example.com:1666')
        self.assertEqual(repository_info.base_path, '')
        self.assertEqual(repository_info.local_path, self.checkout_dir)
        self.assertEqual(client._type, client.TYPE_GIT_P4)

    def test_diff(self) -> None:
        """Testing GitClient.diff with git-p4"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()

        # Pre-cache this.
        client._supports_git_config_flag()

        base_commit_id = self._git_get_head()

        with open('new-file.txt', 'wb') as f:
            f.write(FOO1)

        with open('existing-file.txt', 'ab') as f:
            f.write(b'Here is a new line.\n')

        self._run_git(['add', 'new-file.txt', 'existing-file.txt'])
        self._run_git([
            'commit', '-m',
            ('Set up files for the diff.\n'
             '\n'
             '[git-p4: depot-paths = "//depot/": change = 6]\n'),
        ])

        commit_id = self._git_get_head()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'--- //depot/existing-file.txt\t'
                    b'//depot/existing-file.txt#1\n'
                    b'+++ //depot/existing-file.txt\tTIMESTAMP\n'
                    b'@@ -9,3 +9,4 @@ inferretque deos Latio, genus unde '
                    b'Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b' \n'
                    b'+Here is a new line.\n'
                    b'--- //depot/new-file.txt\t//depot/new-file.txt#1\n'
                    b'+++ //depot/new-file.txt\tTIMESTAMP\n'
                    b'@@ -0,0 +1,9 @@\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+Italiam, fato profugus, Laviniaque venit\n'
                    b'+litora, multum ille et terris iactatus et alto\n'
                    b'+vi superum saevae memorem Iunonis ob iram;\n'
                    b'+multa quoque et bello passus, dum conderet urbem,\n'
                    b'+inferretque deos Latio, genus unde Latinum,\n'
                    b'+Albanique patres, atque altae moenia Romae.\n'
                    b'+Musa, mihi causas memora, quo numine laeso,\n'
                    b'+\n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_spaces_in_filename(self) -> None:
        """Testing GitClient.diff with git-p4 with spaces in filename"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()

        # Pre-cache this.
        client._supports_git_config_flag()

        base_commit_id = self._git_get_head()

        self._git_add_file_commit(
            filename='new  file.txt',
            data=FOO2,
            msg=(
                'Add a file to the base clone.\n'
                '\n'
                '[git-p4: depot-paths = "//depot/": change = 6]\n'
            ))

        commit_id = self._git_get_head()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'--- //depot/new  file.txt\t//depot/new  file.txt#1\n'
                    b'+++ //depot/new  file.txt\tTIMESTAMP\n'
                    b'@@ -0,0 +1,11 @@\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+Italiam, fato profugus, Laviniaque venit\n'
                    b'+litora, multum ille et terris iactatus et alto\n'
                    b'+vi superum saevae memorem Iunonis ob iram;\n'
                    b'+multa quoque et bello passus, dum conderet urbem,\n'
                    b'+inferretque deos Latio, genus unde Latinum,\n'
                    b'+Albanique patres, atque altae moenia Romae.\n'
                    b'+Musa, mihi causas memora, quo numine laeso,\n'
                    b'+\n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_rename(self) -> None:
        """Testing GitClient.diff with renamed file"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()

        # Pre-cache this.
        client._supports_git_config_flag()

        base_commit_id = self._git_get_head()

        self._run_git(['mv', 'existing-file.txt', 'renamed-file.txt'])
        self._run_git(['commit', '-m', 'Rename test.'])

        commit_id = self._git_get_head()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'==== //depot/existing-file.txt#1 ==MV== '
                    b'//depot/renamed-file.txt ====\n'
                    b'\n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_rename_and_changes(self) -> None:
        """Testing GitClient.diff with renamed file and changes"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()

        # Pre-cache this.
        client._supports_git_config_flag()

        base_commit_id = self._git_get_head()

        self._run_git(['mv', 'existing-file.txt', 'renamed-file.txt'])

        with open('renamed-file.txt', 'ab') as fp:
            fp.write(b'Here is a new line!\n')

        self._run_git(['add', 'renamed-file.txt'])
        self._run_git(['commit', '-m', 'Rename test.'])

        commit_id = self._git_get_head()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'Moved from: //depot/existing-file.txt\n'
                    b'Moved to: //depot/renamed-file.txt\n'
                    b'--- //depot/existing-file.txt\t'
                    b'//depot/existing-file.txt#1\n'
                    b'+++ //depot/renamed-file.txt\tTIMESTAMP\n'
                    b'@@ -9,3 +9,4 @@ inferretque deos Latio, genus unde '
                    b'Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b' \n'
                    b'+Here is a new line!\n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_deletes(self) -> None:
        """Testing GitClient.diff with deleted files"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()

        # Pre-cache this.
        client._supports_git_config_flag()

        base_commit_id = self._git_get_head()

        self._run_git(['rm', 'existing-file.txt'])
        self._run_git(['commit', '-m', 'Delete test.'])

        commit_id = self._git_get_head()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'--- //depot/existing-file.txt\t'
                    b'//depot/existing-file.txt#1\n'
                    b'+++ //depot/existing-file.txt\tTIMESTAMP\n'
                    b'@@ -1,11 +0,0 @@\n'
                    b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'-Italiam, fato profugus, Laviniaque venit\n'
                    b'-litora, multum ille et terris iactatus et alto\n'
                    b'-vi superum saevae memorem Iunonis ob iram;\n'
                    b'-multa quoque et bello passus, dum conderet urbem,\n'
                    b'-inferretque deos Latio, genus unde Latinum,\n'
                    b'-Albanique patres, atque altae moenia Romae.\n'
                    b'-Musa, mihi causas memora, quo numine laeso,\n'
                    b'-\n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_multiple_commits(self) -> None:
        """Testing GitClient.diff with git-p4 and multiple commits"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()

        base_commit_id = self._git_get_head()

        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        self._git_add_file_commit('foo.txt', FOO2, 'commit 1')
        self._git_add_file_commit('foo.txt', FOO3, 'commit 1')
        commit_id = self._git_get_head()

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'base_commit_id': base_commit_id,
                'commit_id': commit_id,
                'diff': (
                    b'--- //depot/foo.txt\t//depot/foo.txt#1\n'
                    b'+++ //depot/foo.txt\tTIMESTAMP\n'
                    b'@@ -1,12 +1,11 @@\n ARMA virumque cano, Troiae '
                    b'qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' Italiam, fato profugus, Laviniaque venit\n'
                    b' litora, multum ille et terris iactatus et alto\n'
                    b' vi superum saevae memorem Iunonis ob iram;\n'
                    b'-multa quoque et bello passus, dum conderet urbem,\n'
                    b'+dum conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b'+Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_exclude_patterns(self) -> None:
        """Testing GitClient.diff with git-p4 and file exclusion"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()
        base_commit_id = self._git_get_head()

        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        self._git_add_file_commit('exclude.txt', FOO2, 'commit 2')
        commit_id = self._git_get_head()

        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions, exclude_patterns=['exclude.txt']),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'--- //depot/foo.txt\t//depot/foo.txt#1\n'
                    b'+++ //depot/foo.txt\tTIMESTAMP\n'
                    b'@@ -6,7 +6,4 @@ multa quoque et bello passus, dum '
                    b'conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_exclude_in_subdir(self) -> None:
        """Testing GitClient simple diff with file exclusion in a subdir"""
        client = self.build_client(needs_diff=True)
        base_commit_id = self._git_get_head()

        os.mkdir('subdir')
        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        self._git_add_file_commit('subdir/exclude.txt', FOO2, 'commit 2')

        os.chdir('subdir')
        client.get_repository_info()

        commit_id = self._git_get_head()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions, exclude_patterns=['exclude.txt']),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'--- //depot/foo.txt\t//depot/foo.txt#1\n'
                    b'+++ //depot/foo.txt\tTIMESTAMP\n'
                    b'@@ -6,7 +6,4 @@ multa quoque et bello passus, dum '
                    b'conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_exclude_patterns_root_pattern_in_subdir(self) -> None:
        """Testing GitClient diff with file exclusion in the repo root"""
        client = self.build_client(needs_diff=True)
        base_commit_id = self._git_get_head()

        os.mkdir('subdir')
        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        self._git_add_file_commit('exclude.txt', FOO2, 'commit 2')
        os.chdir('subdir')

        client.get_repository_info()

        commit_id = self._git_get_head()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions,
                        exclude_patterns=[os.path.sep + 'exclude.txt']),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'--- //depot/foo.txt\t//depot/foo.txt#1\n'
                    b'+++ //depot/foo.txt\tTIMESTAMP\n'
                    b'@@ -6,7 +6,4 @@ multa quoque et bello passus, dum '
                    b'conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'-quidve dolens, regina deum tot volvere casus\n'
                    b'-insignem pietate virum, tot adire labores\n'
                    b'-impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                ),
                'parent_diff': None,
            })


class GitSubversionClientTests(BaseGitClientTests):
    """Unit tests for GitClient wrapping Subversion.

    Version Added:
        4.0
    """

    #: The path to the upstream SVN repository.
    svn_repo_path: ClassVar[str]

    @classmethod
    def setup_checkout(
        cls,
        checkout_dir: str,
    ) -> str | None:
        """Populate a Git-SVN checkout.

        This will create a checkout of the sample Git repository stored
        in the :file:`testdata` directory, along with a child clone and a
        grandchild clone.

        Args:
            checkout_dir (str):
                The top-level directory in which the clones will be placed.

        Returns:
            The main checkout directory, or ``None`` if :command:`git` isn't
            in the path.
        """
        clone_dir = super().setup_checkout(checkout_dir)

        if clone_dir is None:
            return None

        svn_clone_dir = os.path.join(clone_dir, 'svn-clone')

        svn_repo_dir = os.path.join(cls.testdata_dir, 'svn-repo')
        cls.svn_repo_path = f'file://{svn_repo_dir}'

        cls._run_git(['svn', 'clone', cls.svn_repo_path, svn_clone_dir])
        os.chdir(svn_clone_dir)

        return os.path.realpath(svn_clone_dir)

    def test_get_repository_info(self) -> None:
        """Testing GitClient.get_repository_info with git-svn"""
        client = self.build_client()
        repository_info = client.get_repository_info()
        assert repository_info is not None

        self.assertEqual(repository_info.path, self.svn_repo_path)
        self.assertEqual(repository_info.base_path, '/')
        self.assertEqual(repository_info.local_path, self.checkout_dir)
        self.assertEqual(client._type, client.TYPE_GIT_SVN)

    def test_parse_revision_spec_no_args(self) -> None:
        """Testing GitClient.parse_revision_spec with git-svn and no
        specified revisions
        """
        client = self.build_client(needs_diff=True)

        base_commit_id = self._git_get_head()
        self._git_add_file_commit('foo.txt', FOO2, 'Commit 2')
        tip_commit_id = self._git_get_head()

        client.get_repository_info()

        self.assertEqual(
            client.parse_revision_spec([]),
            {
                'base': base_commit_id,
                'commit_id': tip_commit_id,
                'tip': tip_commit_id,
            })

    def test_diff(self) -> None:
        """Testing GitClient.diff with git-svn"""
        client = self.build_client(needs_diff=True)
        client.get_repository_info()

        base_commit_id = self._git_get_head()

        with open('new-file.txt', 'wb') as f:
            f.write(FOO1)

        with open('foo.txt', 'ab') as f:
            f.write(b'Here is a new line.\n')

        self._run_git(['add', 'new-file.txt', 'foo.txt'])
        self._run_git([
            'commit', '-m',
            'Set up files for the diff.\n',
        ])

        commit_id = self._git_get_head()
        revisions = client.parse_revision_spec(['HEAD'])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'Index: foo.txt\n'
                    b'===================================================='
                    b'===============\n'
                    b'--- foo.txt\t(revision 7)\n'
                    b'+++ foo.txt\t(working copy)\n'
                    b'@@ -9,3 +9,4 @@ Albanique patres, atque altae moenia '
                    b'Romae.\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b' \n'
                    b'+Here is a new line.\n'
                    b'Index: new-file.txt\n'
                    b'===================================================='
                    b'===============\n'
                    b'--- new-file.txt\t(nonexistent)\n'
                    b'+++ new-file.txt\t(working copy)\n'
                    b'@@ -0,0 +1,9 @@\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+Italiam, fato profugus, Laviniaque venit\n'
                    b'+litora, multum ille et terris iactatus et alto\n'
                    b'+vi superum saevae memorem Iunonis ob iram;\n'
                    b'+multa quoque et bello passus, dum conderet urbem,\n'
                    b'+inferretque deos Latio, genus unde Latinum,\n'
                    b'+Albanique patres, atque altae moenia Romae.\n'
                    b'+Musa, mihi causas memora, quo numine laeso,\n'
                    b'+\n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_spaces_in_filename(self) -> None:
        """Testing GitClient.diff with git-svn with spaces in filename"""
        client = self.build_client(needs_diff=True)

        base_commit_id = self._git_get_head()

        self._git_add_file_commit(
            filename='new  file.txt',
            data=FOO2,
            msg='Add a file to the base clone.')

        commit_id = self._git_get_head()

        client.get_repository_info()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'Index: new  file.txt\n'
                    b'===================================================='
                    b'===============\n'
                    b'--- new  file.txt\t(nonexistent)\n'
                    b'+++ new  file.txt\t(working copy)\n'
                    b'@@ -0,0 +1,11 @@\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+Italiam, fato profugus, Laviniaque venit\n'
                    b'+litora, multum ille et terris iactatus et alto\n'
                    b'+vi superum saevae memorem Iunonis ob iram;\n'
                    b'+multa quoque et bello passus, dum conderet urbem,\n'
                    b'+inferretque deos Latio, genus unde Latinum,\n'
                    b'+Albanique patres, atque altae moenia Romae.\n'
                    b'+Musa, mihi causas memora, quo numine laeso,\n'
                    b'+\n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_deletes(self) -> None:
        """Testing GitClient.diff with git-svn and deleted files"""
        client = self.build_client(needs_diff=True)

        base_commit_id = self._git_get_head()

        self._run_git(['rm', 'foo.txt'])
        self._run_git(['commit', '-m', 'Delete test.'])

        commit_id = self._git_get_head()

        client.get_repository_info()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'Index: foo.txt\n'
                    b'===================================================='
                    b'===============\n'
                    b'--- foo.txt\t(revision 7)\n'
                    b'+++ foo.txt\t(nonexistent)\n'
                    b'@@ -1,11 +0,0 @@\n'
                    b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'-Italiam, fato profugus, Laviniaque venit\n'
                    b'-litora, multum ille et terris iactatus et alto\n'
                    b'-vi superum saevae memorem Iunonis ob iram;\n'
                    b'-dum conderet urbem,\n'
                    b'-inferretque deos Latio, genus unde Latinum,\n'
                    b'-Albanique patres, atque altae moenia Romae.\n'
                    b'-Albanique patres, atque altae moenia Romae.\n'
                    b'-Musa, mihi causas memora, quo numine laeso,\n'
                    b'-\n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_multiple_commits(self) -> None:
        """Testing GitClient.diff with git-svn and multiple commits"""
        client = self.build_client(needs_diff=True)

        base_commit_id = self._git_get_head()

        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        self._git_add_file_commit('foo.txt', FOO2, 'commit 2')

        commit_id = self._git_get_head()

        client.get_repository_info()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions),
            {
                'base_commit_id': base_commit_id,
                'commit_id': commit_id,
                'diff': (
                    b'Index: foo.txt\n'
                    b'===================================================='
                    b'===============\n'
                    b'--- foo.txt\t(revision 7)\n'
                    b'+++ foo.txt\t(working copy)\n'
                    b'@@ -1,11 +1,11 @@\n'
                    b' ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' Italiam, fato profugus, Laviniaque venit\n'
                    b' litora, multum ille et terris iactatus et alto\n'
                    b' vi superum saevae memorem Iunonis ob iram;\n'
                    b'-dum conderet urbem,\n'
                    b'+multa quoque et bello passus, dum conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b'-Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b' \n'
                ),
                'parent_diff': None,
            })

    def test_diff_with_exclude_patterns(self) -> None:
        """Testing GitClient.diff with git-svn and file exclusion"""
        client = self.build_client(needs_diff=True)
        base_commit_id = self._git_get_head()

        self._git_add_file_commit('foo.txt', FOO1, 'commit 1')
        self._git_add_file_commit('exclude.txt', FOO2, 'commit 2')
        commit_id = self._git_get_head()

        client.get_repository_info()
        revisions = client.parse_revision_spec([])

        self.assertEqual(
            client.diff(revisions, exclude_patterns=['exclude.txt']),
            {
                'commit_id': commit_id,
                'base_commit_id': base_commit_id,
                'diff': (
                    b'Index: foo.txt\n'
                    b'===================================================='
                    b'===============\n'
                    b'--- foo.txt\t(revision 7)\n'
                    b'+++ foo.txt\t(working copy)\n'
                    b'@@ -1,11 +1,9 @@\n'
                    b' ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' Italiam, fato profugus, Laviniaque venit\n'
                    b' litora, multum ille et terris iactatus et alto\n'
                    b' vi superum saevae memorem Iunonis ob iram;\n'
                    b'-dum conderet urbem,\n'
                    b'+multa quoque et bello passus, dum conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b'-Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b' \n'
                ),
                'parent_diff': None,
            })


class GitPatcherTests(BaseGitClientTests):
    """Unit tests for GitPatcher.

    Version Added:
        5.1
    """

    @classmethod
    def setup_checkout(
        cls,
        checkout_dir: str,
    ) -> str | None:
        """Populate a Git checkout.

        This will create a checkout of the sample Git repository stored
        in the :file:`testdata` directory, along with a child clone and a
        grandchild clone.

        Args:
            checkout_dir (str):
                The top-level directory in which the clones will be placed.

        Returns:
            str:
            The main checkout directory, or ``None`` if :command:`git` isn't
            in the path.
        """
        clone_dir = super().setup_checkout(checkout_dir)

        if clone_dir is None:
            return None

        orig_clone_dir = os.path.join(checkout_dir, 'orig')
        cls._run_git(['clone', cls.git_dir, orig_clone_dir])

        return orig_clone_dir

    def test_patch(self) -> None:
        """Testing GitPatcher.patch"""
        client = self.build_client()

        # Refresh state so that indexes will be looked up. For some reason,
        # we need to do this after building the client.
        self._run_git(['update-index', '--refresh'])

        head = self._git_get_head()

        patcher = client.get_patcher(patches=[
            Patch(content=(
                b'diff --git a/foo.txt b/foo.txt\n'
                b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -6,7 +6,4 @@ multa quoque et bello passus, '
                b'dum conderet urbem,\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            )),
        ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', mode='rb') as fp:
            self.assertEqual(fp.read(), FOO1)

        # There should not be a new commit. HEAD won't change.
        self.assertEqual(self._git_get_head(), head)

    def test_patch_with_commit(self) -> None:
        """Testing GitPatcher.patch with committing"""
        client = self.build_client()

        # Refresh state so that indexes will be looked up. For some reason,
        # we need to do this after building the client.
        self._run_git(['update-index', '--refresh'])

        num_commits = self._git_get_num_commits()

        patcher = client.get_patcher(patches=[
            Patch(content=(
                b'diff --git a/foo.txt b/foo.txt\n'
                b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -6,7 +6,4 @@ multa quoque et bello passus, '
                b'dum conderet urbem,\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            )),
        ])
        patcher.prepare_for_commit(
            default_author=PatchAuthor(full_name='Test User',
                                       email='test@example.com'),
            default_message='Test message')

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', mode='rb') as fp:
            self.assertEqual(fp.read(), FOO1)

        # There should be a new commit.
        self.assertEqual(self._git_get_num_commits(), num_commits + 1)

    def test_patch_with_multiple_patches(self) -> None:
        """Testing GitPatcher.patch with multiple patches"""
        client = self.build_client()

        # Refresh state so that indexes will be looked up. For some reason,
        # we need to do this after building the client.
        self._run_git(['update-index', '--refresh'])

        num_commits = self._git_get_num_commits()

        patcher = client.get_patcher(patches=[
            Patch(content=(
                b'diff --git a/foo.txt b/foo.txt\n'
                b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -6,7 +6,4 @@ multa quoque et bello passus, '
                b'dum conderet urbem,\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            )),
            Patch(content=(
                b'diff --git a/foo.txt b/foo.txt\n'
                b'index 5e98e9540e1b741b5be24fcb33c40c1c8069c1fb..'
                b'e619c1387f5feb91f0ca83194650bfe4f6c2e347 100644\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -1,4 +1,6 @@\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b' Italiam, fato profugus, Laviniaque venit\n'
                b' litora, multum ille et terris iactatus et alto\n'
                b' vi superum saevae memorem Iunonis ob iram;\n'
            )),
        ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 2)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        result = results[1]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', mode='rb') as fp:
            self.assertEqual(fp.read(), FOO2)

        # There should not be a new commit. HEAD won't change.
        self.assertEqual(self._git_get_num_commits(), num_commits)

    def test_patch_with_multiple_patches_and_commit(self) -> None:
        """Testing GitPatcher.patch with multiple patches and committing"""
        client = self.build_client()

        # Refresh state so that indexes will be looked up. For some reason,
        # we need to do this after building the client.
        self._run_git(['update-index', '--refresh'])

        num_commits = self._git_get_num_commits()

        patcher = client.get_patcher(patches=[
            Patch(content=(
                b'diff --git a/foo.txt b/foo.txt\n'
                b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -6,7 +6,4 @@ multa quoque et bello passus, '
                b'dum conderet urbem,\n'
                b' inferretque deos Latio, genus unde Latinum,\n'
                b' Albanique patres, atque altae moenia Romae.\n'
                b' Musa, mihi causas memora, quo numine laeso,\n'
                b'-quidve dolens, regina deum tot volvere casus\n'
                b'-insignem pietate virum, tot adire labores\n'
                b'-impulerit. Tantaene animis caelestibus irae?\n'
                b' \n'
            )),
            Patch(content=(
                b'diff --git a/foo.txt b/foo.txt\n'
                b'index 5e98e9540e1b741b5be24fcb33c40c1c8069c1fb..'
                b'e619c1387f5feb91f0ca83194650bfe4f6c2e347 100644\n'
                b'--- a/foo.txt\n'
                b'+++ b/foo.txt\n'
                b'@@ -1,4 +1,6 @@\n'
                b' ARMA virumque cano, Troiae qui primus ab oris\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b'+ARMA virumque cano, Troiae qui primus ab oris\n'
                b' Italiam, fato profugus, Laviniaque venit\n'
                b' litora, multum ille et terris iactatus et alto\n'
                b' vi superum saevae memorem Iunonis ob iram;\n'
            )),
        ])
        patcher.prepare_for_commit(
            default_author=PatchAuthor(full_name='Test User',
                                       email='test@example.com'),
            default_message='Test message')

        results = list(patcher.patch())
        self.assertEqual(len(results), 2)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        result = results[1]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', mode='rb') as fp:
            self.assertEqual(fp.read(), FOO2)

        # There should be two new commits.
        self.assertEqual(self._git_get_num_commits(), num_commits + 2)

    def test_patch_with_revert(self) -> None:
        """Testing GitPatcher.patch with revert"""
        client = self.build_client()

        # Refresh state so that indexes will be looked up. For some reason,
        # we need to do this after building the client.
        self._run_git(['update-index', '--refresh'])

        num_commits = self._git_get_num_commits()

        patcher = client.get_patcher(
            revert=True,
            patches=[
                Patch(content=(
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,4 +6,7 @@ multa quoque et bello passus, '
                    b'dum conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'+quidve dolens, regina deum tot volvere casus\n'
                    b'+insignem pietate virum, tot adire labores\n'
                    b'+impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                )),
                Patch(content=(
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 5e98e9540e1b741b5be24fcb33c40c1c8069c1fb..'
                    b'e619c1387f5feb91f0ca83194650bfe4f6c2e347 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -1,6 +1,4 @@\n'
                    b' ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' Italiam, fato profugus, Laviniaque venit\n'
                    b' litora, multum ille et terris iactatus et alto\n'
                    b' vi superum saevae memorem Iunonis ob iram;\n'
                )),
            ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 2)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        result = results[1]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', mode='rb') as fp:
            self.assertEqual(
                fp.read(),
                b'ARMA virumque cano, Troiae qui primus ab oris\n'
                b'ARMA virumque cano, Troiae qui primus ab oris\n'
                b'ARMA virumque cano, Troiae qui primus ab oris\n'
                b'Italiam, fato profugus, Laviniaque venit\n'
                b'litora, multum ille et terris iactatus et alto\n'
                b'vi superum saevae memorem Iunonis ob iram;\n'
                b'multa quoque et bello passus, dum conderet urbem,\n'
                b'inferretque deos Latio, genus unde Latinum,\n'
                b'Albanique patres, atque altae moenia Romae.\n'
                b'Musa, mihi causas memora, quo numine laeso,\n'
                b'\n')

        # There should not be a new commit. HEAD won't change.
        self.assertEqual(self._git_get_num_commits(), num_commits)

    def test_patch_with_revert_and_commit(self) -> None:
        """Testing GitPatcher.patch with revert and commit"""
        client = self.build_client()

        # Refresh state so that indexes will be looked up. For some reason,
        # we need to do this after building the client.
        self._run_git(['update-index', '--refresh'])

        num_commits = self._git_get_num_commits()

        patcher = client.get_patcher(
            revert=True,
            patches=[
                Patch(content=(
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
                    b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -6,4 +6,7 @@ multa quoque et bello passus, '
                    b'dum conderet urbem,\n'
                    b' inferretque deos Latio, genus unde Latinum,\n'
                    b' Albanique patres, atque altae moenia Romae.\n'
                    b' Musa, mihi causas memora, quo numine laeso,\n'
                    b'+quidve dolens, regina deum tot volvere casus\n'
                    b'+insignem pietate virum, tot adire labores\n'
                    b'+impulerit. Tantaene animis caelestibus irae?\n'
                    b' \n'
                )),
                Patch(content=(
                    b'diff --git a/foo.txt b/foo.txt\n'
                    b'index 5e98e9540e1b741b5be24fcb33c40c1c8069c1fb..'
                    b'e619c1387f5feb91f0ca83194650bfe4f6c2e347 100644\n'
                    b'--- a/foo.txt\n'
                    b'+++ b/foo.txt\n'
                    b'@@ -1,6 +1,4 @@\n'
                    b' ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                    b'-ARMA virumque cano, Troiae qui primus ab oris\n'
                    b' Italiam, fato profugus, Laviniaque venit\n'
                    b' litora, multum ille et terris iactatus et alto\n'
                    b' vi superum saevae memorem Iunonis ob iram;\n'
                )),
            ])
        patcher.prepare_for_commit(
            default_author=PatchAuthor(full_name='Test User',
                                       email='test@example.com'),
            default_message='Test message')

        results = list(patcher.patch())
        self.assertEqual(len(results), 2)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        result = results[1]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', mode='rb') as fp:
            self.assertEqual(
                fp.read(),
                b'ARMA virumque cano, Troiae qui primus ab oris\n'
                b'ARMA virumque cano, Troiae qui primus ab oris\n'
                b'ARMA virumque cano, Troiae qui primus ab oris\n'
                b'Italiam, fato profugus, Laviniaque venit\n'
                b'litora, multum ille et terris iactatus et alto\n'
                b'vi superum saevae memorem Iunonis ob iram;\n'
                b'multa quoque et bello passus, dum conderet urbem,\n'
                b'inferretque deos Latio, genus unde Latinum,\n'
                b'Albanique patres, atque altae moenia Romae.\n'
                b'Musa, mihi causas memora, quo numine laeso,\n'
                b'\n')

        # There should be two new commits.
        self.assertEqual(self._git_get_num_commits(), num_commits + 2)

    def test_binary_file_add(self) -> None:
        """Testing GitPatcher with an added binary file"""
        client = self.build_client()

        test_content = b'Binary file content'
        test_path = 'new_binary_file.bin'

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 123,
                'absolute_url': 'https://example.com/r/1/file/123/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/123/'
        )

        binary_file = self.make_binary_file_patch(
            old_path=None,
            new_path=test_path,
            status='added',
            file_attachment=attachment,
            content=test_content,
        )

        patch_content = (
            b'diff --git a/new_binary_file.bin b/new_binary_file.bin\n'
            b'new file mode 100644\n'
            b'index 0000000..e69de29 100644\n'
            b'Binary files /dev/null and b/new_binary_file.bin differ\n'
        )
        patch = Patch(content=patch_content, binary_files=[binary_file])
        patcher = client.get_patcher(patches=[patch])

        self.spy_on(client._run_git)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], test_path)

        self.assertTrue(os.path.exists(test_path))

        with open(test_path, mode='rb') as f:
            self.assertEqual(f.read(), test_content)

        self.assertSpyCalledWith(
            client._run_git, ['add', f':(literal){test_path}'])

    def test_binary_file_add_with_special_chars(self) -> None:
        """Testing GitPatcher with binary file containing special characters"""
        client = self.build_client()

        test_content = b'Binary file content'
        # Test path with special characters that need escaping: *, ?, [
        test_path = 'images/test_file*with?special[chars].bin'

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 123,
                'absolute_url': 'https://example.com/r/1/file/123/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/123/'
        )

        binary_file = self.make_binary_file_patch(
            old_path=None,
            new_path=test_path,
            status='added',
            file_attachment=attachment,
            content=test_content,
        )

        patch_content = (
            b'diff --git a/images/test_file*with?special[chars].bin '
            b'b/images/test_file*with?special[chars].bin\n'
            b'new file mode 100644\n'
            b'index 0000000..e69de29 100644\n'
            b'Binary files /dev/null and '
            b'b/images/test_file*with?special[chars].bin differ\n'
        )
        patch = Patch(content=patch_content, binary_files=[binary_file])
        patcher = client.get_patcher(patches=[patch])

        self.spy_on(client._run_git)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], test_path)

        self.assertTrue(os.path.exists(test_path))

        with open(test_path, mode='rb') as f:
            self.assertEqual(f.read(), test_content)

        self.assertSpyCalledWith(
            client._run_git, ['add', f':(literal){test_path}'])

        # Verify that the path was properly escaped in the git apply call
        # The escaped path should have *, ?, [ characters escaped with
        # backslashes.
        git_apply_calls = [
            call for call in client._run_git.spy.calls
            if call.args[0][0] == 'apply'
        ]
        self.assertEqual(len(git_apply_calls), 1)

        apply_command = git_apply_calls[0].args[0]

        # Check that the exclude parameter has properly escaped special
        # characters.
        exclude_arg = None

        for arg in apply_command:
            if arg.startswith('--exclude='):
                exclude_arg = arg
                break

        self.assertIsNotNone(exclude_arg)
        self.assertEqual(
            exclude_arg,
            '--exclude=images/test_file\\*with\\?special\\[chars].bin')

        with open(test_path, mode='rb') as f:
            self.assertEqual(f.read(), test_content)

    def test_binary_file_add_in_subdirectory(self) -> None:
        """Testing GitPatcher with an added binary file in a subdirectory"""
        client = self.build_client()

        test_content = b'Binary file content'
        test_path = 'subdir/new_binary_file.bin'

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 124,
                'absolute_url': 'https://example.com/r/1/file/124/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/124/'
        )

        binary_file = self.make_binary_file_patch(
            old_path=None,
            new_path=test_path,
            status='added',
            file_attachment=attachment,
            content=test_content,
        )

        patch_content = (
            b'diff --git a/subdir/new_binary_file.bin '
            b'b/subdir/new_binary_file.bin\n'
            b'new file mode 100644\n'
            b'index 0000000000000000000000000000000000000000..'
            b'0a0efb879d12d3480ad13cfb2e5db80db8f52ea1\n'
            b'Binary files /dev/null and b/subdir/new_binary_file.bin differ\n'
        )
        patch = Patch(content=patch_content, binary_files=[binary_file])
        patcher = client.get_patcher(patches=[patch])

        self.spy_on(client._run_git)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], test_path)

        self.assertTrue(os.path.exists('subdir'))
        self.assertTrue(os.path.exists(test_path))

        with open(test_path, mode='rb') as f:
            self.assertEqual(f.read(), test_content)

        self.assertSpyCalledWith(
            client._run_git, ['add', f':(literal){test_path}'])

    def test_binary_file_move(self) -> None:
        """Testing GitPatcher with a moved binary file"""
        client = self.build_client()

        old_path = 'old_file.bin'
        new_path = 'new_file.bin'
        test_content = b'Binary file content'

        with open(old_path, mode='wb') as f:
            f.write(b'Old file content')

        self._run_git(['add', old_path])
        self._run_git(['commit', '-m', 'Add old file'])

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 125,
                'absolute_url': 'https://example.com/r/1/file/125/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/125/'
        )

        binary_file = self.make_binary_file_patch(
            old_path=old_path,
            new_path=new_path,
            status='moved',
            file_attachment=attachment,
            content=test_content,
        )

        patch_content = (
            b'diff --git a/old_file.bin b/new_file.bin\n'
            b'index c9385c77b66550a9abc0bacc5fd7d739e74c12bd..'
            b'0a0efb879d12d3480ad13cfb2e5db80db8f52ea1 100644\n'
            b'rename from old_file.bin\n'
            b'rename to new_file.bin\n'
            b'Binary files a/old_file.bin and b/new_file.bin differ\n'
        )
        patch = Patch(content=patch_content, binary_files=[binary_file])
        patcher = client.get_patcher(patches=[patch])

        self.spy_on(client._run_git)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], new_path)

        self.assertSpyCalledWith(client._run_git, ['mv', old_path, new_path])

    def test_binary_file_move_with_subdirectory(self) -> None:
        """Testing GitPatcher with a moved binary file into a subdirectory"""
        client = self.build_client()

        old_path = 'old_file.bin'
        new_path = 'subdir/new_file.bin'
        test_content = b'Binary file content'

        with open(old_path, mode='wb') as f:
            f.write(b'Old file content')

        self._run_git(['add', old_path])

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 126,
                'absolute_url': 'https://example.com/r/1/file/126/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/126/'
        )

        binary_file = self.make_binary_file_patch(
            old_path=old_path,
            new_path=new_path,
            status='moved',
            file_attachment=attachment,
            content=test_content,
        )

        patch_content = (
            b'diff --git a/old_file.bin b/subdir/new_file.bin\n'
            b'index c9385c77b66550a9abc0bacc5fd7d739e74c12bd..'
            b'0a0efb879d12d3480ad13cfb2e5db80db8f52ea1 100644\n'
            b'rename from old_file.bin\n'
            b'rename to subdir/new_file.bin\n'
            b'Binary files a/old_file.bin and b/subdir/new_file.bin differ\n'
        )
        patch = Patch(content=patch_content, binary_files=[binary_file])
        patcher = client.get_patcher(patches=[patch])

        self.spy_on(client._run_git)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], new_path)

        self.assertTrue(os.path.exists('subdir'))

        self.assertSpyCalledWith(client._run_git, ['mv', old_path, new_path])

    def test_binary_file_remove(self) -> None:
        """Testing GitPatcher with a removed binary file."""
        client = self.build_client()

        test_path = 'file_to_remove.bin'
        test_content = b'Binary file content'

        with open(test_path, mode='wb') as f:
            f.write(test_content)

        self._run_git(['add', test_path])
        self._run_git(['commit', '-m', 'Add test file'])

        binary_file = self.make_binary_file_patch(
            old_path=test_path,
            new_path=None,
            status='deleted',
            file_attachment=None,
            content=test_content,
        )

        patch_content = (
            b'diff --git a/dummy b/dummy\n'
            b'new file mode 100644\n'
            b'index 0000000..e69de29\n'
            b'--- /dev/null\n'
            b'+++ b/dummy\n'
        )
        patch = Patch(content=patch_content, binary_files=[binary_file])
        patcher = client.get_patcher(patches=[patch])

        self.spy_on(client._run_git)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], test_path)

        self.assertSpyCalledWith(
            client._run_git, ['rm', f':(literal){test_path}'])

    def test_patch_with_empty_files(self) -> None:
        """Testing GitPatcher.patch with empty files."""
        client = self.build_client()

        empty_to_delete = 'empty_delete.txt'
        with open(empty_to_delete, mode='w', encoding='utf-8') as f:
            pass

        self._run_git(['add', empty_to_delete])
        self._run_git(['commit', '-m', 'Add empty file'])

        empty_to_add = 'empty_add.txt'

        patch_content = (
            b'diff --git a/empty_add.txt b/empty_add.txt\n'
            b'new file mode 100644\n'
            b'index 0000000..e69de29\n'
            b'--- /dev/null\n'
            b'+++ b/empty_add.txt\n'
            b'diff --git a/empty_delete.txt b/empty_delete.txt\n'
            b'deleted file mode 100644\n'
            b'index e69de29..0000000\n'
            b'--- a/empty_delete.txt\n'
            b'+++ /dev/null\n'
        )

        patcher = client.get_patcher(patches=[
            Patch(content=patch_content),
        ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        self.assertTrue(os.path.exists(empty_to_add))

        with open(empty_to_add, mode='rb') as f:
            self.assertEqual(f.read(), b'')

        self.assertFalse(os.path.exists(empty_to_delete))

    def test_patch_with_regular_and_empty_files(self) -> None:
        """Testing GitPatcher.patch with regular and empty files."""
        client = self.build_client()

        self._run_git(['update-index', '--refresh'])

        empty_to_delete = 'empty_delete.txt'
        with open(empty_to_delete, mode='w', encoding='utf-8') as f:
            pass

        self._run_git(['add', empty_to_delete])
        self._run_git(['commit', '-m', 'Add empty file'])

        empty_to_add = 'empty_add.txt'

        patch_content = (
            b'diff --git a/foo.txt b/foo.txt\n'
            b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
            b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
            b'--- a/foo.txt\n'
            b'+++ b/foo.txt\n'
            b'@@ -6,7 +6,4 @@ multa quoque et bello passus, '
            b'dum conderet urbem,\n'
            b' inferretque deos Latio, genus unde Latinum,\n'
            b' Albanique patres, atque altae moenia Romae.\n'
            b' Musa, mihi causas memora, quo numine laeso,\n'
            b'-quidve dolens, regina deum tot volvere casus\n'
            b'-insignem pietate virum, tot adire labores\n'
            b'-impulerit. Tantaene animis caelestibus irae?\n'
            b' \n'
            b'diff --git a/empty_add.txt b/empty_add.txt\n'
            b'new file mode 100644\n'
            b'index 0000000..e69de29\n'
            b'--- /dev/null\n'
            b'+++ b/empty_add.txt\n'
            b'diff --git a/empty_delete.txt b/empty_delete.txt\n'
            b'deleted file mode 100644\n'
            b'index e69de29..0000000\n'
            b'--- a/empty_delete.txt\n'
            b'+++ /dev/null\n'
        )

        patcher = client.get_patcher(patches=[
            Patch(content=patch_content),
        ])

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertIsNotNone(result.patch)

        with open('foo.txt', mode='rb') as fp:
            self.assertEqual(fp.read(), FOO1)

        self.assertTrue(os.path.exists(empty_to_add))
        with open(empty_to_add, mode='rb') as f:
            self.assertEqual(f.read(), b'')

        self.assertFalse(os.path.exists(empty_to_delete))

    def test_patch_with_regular_and_binary_files(self) -> None:
        """Testing GitPatcher.patch with regular and binary files."""
        client = self.build_client()

        self._run_git(['update-index', '--refresh'])

        binary_add_path = 'new_image.png'
        binary_add_content = b'\x89PNG\r\n\x1a\n new image'

        binary_modify_path = 'existing.bin'
        binary_modify_old_content = b'old binary content'
        binary_modify_new_content = b'new binary content'

        with open(binary_modify_path, mode='wb') as f:
            f.write(binary_modify_old_content)

        self._run_git(['add', binary_modify_path])
        self._run_git(['commit', '-m', 'Add existing binary'])

        attachment_add = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 201,
                'absolute_url': 'https://example.com/r/1/file/201/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/201/'
        )

        attachment_modify = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 202,
                'absolute_url': 'https://example.com/r/1/file/202/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/202/'
        )

        binary_file_add = self.make_binary_file_patch(
            old_path=None,
            new_path=binary_add_path,
            status='added',
            file_attachment=attachment_add,
            content=binary_add_content,
        )

        binary_file_modify = self.make_binary_file_patch(
            old_path=binary_modify_path,
            new_path=binary_modify_path,
            status='modified',
            file_attachment=attachment_modify,
            content=binary_modify_new_content,
        )

        patch_content = (
            b'diff --git a/foo.txt b/foo.txt\n'
            b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
            b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
            b'--- a/foo.txt\n'
            b'+++ b/foo.txt\n'
            b'@@ -6,7 +6,4 @@ multa quoque et bello passus, '
            b'dum conderet urbem,\n'
            b' inferretque deos Latio, genus unde Latinum,\n'
            b' Albanique patres, atque altae moenia Romae.\n'
            b' Musa, mihi causas memora, quo numine laeso,\n'
            b'-quidve dolens, regina deum tot volvere casus\n'
            b'-insignem pietate virum, tot adire labores\n'
            b'-impulerit. Tantaene animis caelestibus irae?\n'
            b' \n'
            b'diff --git a/new_image.png b/new_image.png\n'
            b'new file mode 100644\n'
            b'index 0000000..abc1234\n'
            b'Binary files /dev/null and b/new_image.png differ\n'
            b'diff --git a/existing.bin b/existing.bin\n'
            b'index def5678..abc1234 100644\n'
            b'Binary files a/existing.bin and b/existing.bin differ\n'
        )

        patch = Patch(
            content=patch_content,
            binary_files=[binary_file_add, binary_file_modify]
        )
        patcher = client.get_patcher(patches=[patch])

        self.spy_on(client._run_git)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 2)
        self.assertIn(binary_add_path, result.binary_applied)
        self.assertIn(binary_modify_path, result.binary_applied)

        with open('foo.txt', mode='rb') as fp:
            self.assertEqual(fp.read(), FOO1)

        self.assertTrue(os.path.exists(binary_add_path))
        with open(binary_add_path, mode='rb') as f:
            self.assertEqual(f.read(), binary_add_content)

        self.assertTrue(os.path.exists(binary_modify_path))
        with open(binary_modify_path, mode='rb') as f:
            self.assertEqual(f.read(), binary_modify_new_content)

    def test_patch_with_empty_and_binary_files(self) -> None:
        """Testing GitPatcher.patch with empty and binary files."""
        client = self.build_client()

        empty_to_delete = 'empty_delete.txt'
        with open(empty_to_delete, mode='w', encoding='utf-8') as f:
            pass

        self._run_git(['add', empty_to_delete])
        self._run_git(['commit', '-m', 'Add empty file'])

        empty_to_add = 'empty_add.txt'
        binary_path = 'archive.zip'
        binary_content = b'PK\x03\x04 zip content'

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 300,
                'absolute_url': 'https://example.com/r/1/file/300/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/300/'
        )

        binary_file = self.make_binary_file_patch(
            old_path=None,
            new_path=binary_path,
            status='added',
            file_attachment=attachment,
            content=binary_content,
        )

        patch_content = (
            b'diff --git a/empty_add.txt b/empty_add.txt\n'
            b'new file mode 100644\n'
            b'index 0000000..e69de29\n'
            b'--- /dev/null\n'
            b'+++ b/empty_add.txt\n'
            b'diff --git a/empty_delete.txt b/empty_delete.txt\n'
            b'deleted file mode 100644\n'
            b'index e69de29..0000000\n'
            b'--- a/empty_delete.txt\n'
            b'+++ /dev/null\n'
            b'diff --git a/archive.zip b/archive.zip\n'
            b'new file mode 100644\n'
            b'index 0000000..abc1234\n'
            b'Binary files /dev/null and b/archive.zip differ\n'
        )

        patch = Patch(
            content=patch_content,
            binary_files=[binary_file]
        )
        patcher = client.get_patcher(patches=[patch])

        self.spy_on(client._run_git)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 1)
        self.assertEqual(result.binary_applied[0], binary_path)

        self.assertTrue(os.path.exists(empty_to_add))
        with open(empty_to_add, mode='rb') as f:
            self.assertEqual(f.read(), b'')

        self.assertFalse(os.path.exists(empty_to_delete))

        self.assertTrue(os.path.exists(binary_path))
        with open(binary_path, mode='rb') as f:
            self.assertEqual(f.read(), binary_content)

    def test_patch_with_mixed_file_types(self) -> None:
        """Testing GitPatcher.patch with regular, empty, and binary files.

        This test combines all three file types (regular, empty, and binary)
        with various operations in a single patch.
        """
        client = self.build_client()

        self._run_git(['update-index', '--refresh'])

        empty_to_delete = 'empty_delete.txt'
        with open(empty_to_delete, mode='w', encoding='utf-8') as f:
            pass

        binary_to_delete = 'old_file.bin'
        with open(binary_to_delete, mode='wb') as f:
            f.write(b'old binary to delete')

        self._run_git(['add', empty_to_delete, binary_to_delete])
        self._run_git(['commit', '-m', 'Add files to delete'])

        empty_to_add = 'empty_add.txt'
        binary_add_path = 'new_data.bin'
        binary_add_content = b'\x00\x01\x02\x03 new data'

        attachment = FileAttachmentItemResource(
            transport=URLMapTransport('https://reviews.example.com/'),
            payload={
                'id': 400,
                'absolute_url': 'https://example.com/r/1/file/400/download/',
            },
            url='https://reviews.example.com/api/review-requests/1/'
                'file-attachments/400/'
        )

        binary_file_add = self.make_binary_file_patch(
            old_path=None,
            new_path=binary_add_path,
            status='added',
            file_attachment=attachment,
            content=binary_add_content,
        )

        binary_file_delete = BinaryFilePatch(
            old_path=binary_to_delete,
            new_path=None,
            status='deleted',
            file_attachment=None,
        )

        patch_content = (
            b'diff --git a/foo.txt b/foo.txt\n'
            b'index 634b3e8ff85bada6f928841a9f2c505560840b3a..'
            b'5e98e9540e1b741b5be24fcb33c40c1c8069c1fb 100644\n'
            b'--- a/foo.txt\n'
            b'+++ b/foo.txt\n'
            b'@@ -6,7 +6,4 @@ multa quoque et bello passus, '
            b'dum conderet urbem,\n'
            b' inferretque deos Latio, genus unde Latinum,\n'
            b' Albanique patres, atque altae moenia Romae.\n'
            b' Musa, mihi causas memora, quo numine laeso,\n'
            b'-quidve dolens, regina deum tot volvere casus\n'
            b'-insignem pietate virum, tot adire labores\n'
            b'-impulerit. Tantaene animis caelestibus irae?\n'
            b' \n'
            b'diff --git a/empty_add.txt b/empty_add.txt\n'
            b'new file mode 100644\n'
            b'index 0000000..e69de29\n'
            b'--- /dev/null\n'
            b'+++ b/empty_add.txt\n'
            b'diff --git a/empty_delete.txt b/empty_delete.txt\n'
            b'deleted file mode 100644\n'
            b'index e69de29..0000000\n'
            b'--- a/empty_delete.txt\n'
            b'+++ /dev/null\n'
            b'diff --git a/new_data.bin b/new_data.bin\n'
            b'new file mode 100644\n'
            b'index 0000000..abc1234\n'
            b'Binary files /dev/null and b/new_data.bin differ\n'
            b'diff --git a/old_file.bin b/old_file.bin\n'
            b'deleted file mode 100644\n'
            b'index def5678..0000000\n'
            b'Binary files a/old_file.bin and /dev/null differ\n'
        )

        patch = Patch(
            content=patch_content,
            binary_files=[binary_file_add, binary_file_delete]
        )
        patcher = client.get_patcher(patches=[patch])

        self.spy_on(client._run_git)

        results = list(patcher.patch())
        self.assertEqual(len(results), 1)

        result = results[0]
        self.assertTrue(result.success)
        self.assertEqual(len(result.binary_applied), 2)
        self.assertIn(binary_add_path, result.binary_applied)
        self.assertIn(binary_to_delete, result.binary_applied)

        with open('foo.txt', mode='rb') as fp:
            self.assertEqual(fp.read(), FOO1)

        self.assertTrue(os.path.exists(empty_to_add))
        with open(empty_to_add, mode='rb') as f:
            self.assertEqual(f.read(), b'')

        self.assertFalse(os.path.exists(empty_to_delete))

        self.assertTrue(os.path.exists(binary_add_path))
        with open(binary_add_path, mode='rb') as f:
            self.assertEqual(f.read(), binary_add_content)

        self.assertFalse(os.path.exists(binary_to_delete))
