"""Resource base classes for the RBTools Python API.

Version Added:
    6.0:
    This was moved from :py:mod:`rbtools.api.resource`.
"""

from __future__ import annotations

import copy
import json
import logging
from collections.abc import Callable, MutableMapping
from functools import update_wrapper, wraps
from typing import (Any, Generic, Literal, TYPE_CHECKING, TypeVar, cast,
                    overload)
from urllib.parse import urljoin

from typelets.json import JSONDict, JSONValue
from typing_extensions import NotRequired, ParamSpec, Self, TypedDict, Unpack

from rbtools.api.request import HttpRequest, QueryArgs
from rbtools.api.utils import rem_mime_format

if TYPE_CHECKING:
    from collections.abc import Iterator, Mapping
    from typing import ClassVar, Final, NoReturn

    from rbtools.api.transport import Transport


logger = logging.getLogger(__name__)


#: Map from MIME type to resource class.
RESOURCE_MAP: dict[str, type[Resource]] = {}

#: The name of the links structure within a response payload.
LINKS_TOK: Final[str] = 'links'

#: The name of the expanded info structure within a response payload.
EXPANDED_TOKEN: Final[str] = '_expanded'

#: Keys within a link dict.
LINK_KEYS: set[str] = {'href', 'method', 'title', 'mimetype'}

#: Default attributes to exclude when processing a response.
_EXCLUDE_ATTRS: set[str] = {LINKS_TOK, EXPANDED_TOKEN, 'stat'}

#: Prefix for keys which should be stored in extra data.
_EXTRA_DATA_PREFIX: Final[str] = 'extra_data__'

#: URL for documentation on working with extra data.
_EXTRA_DATA_DOCS_URL: Final[str] = (
    'https://www.reviewboard.org/docs/manual/latest/webapi/2.0/extra-data/'
     '#storing-merging-json-data'
 )


_P = ParamSpec('_P')
_T = TypeVar('_T')


def request_method(
    f: Callable[_P, HttpRequest],
) -> Callable[_P, RequestMethodResult]:
    """Wrap a method returned from a resource to capture HttpRequests.

    When a method which returns HttpRequests is called, it will
    pass the method and arguments off to the transport to be executed.

    This wrapping allows the transport to skim arguments off the top
    of the method call, and modify any return values (such as executing
    a returned HttpRequest).

    However, if called with the ``internal`` argument set to True,
    the method itself will be executed and the value returned as-is.
    Thus, any method calls embedded inside the code for another method
    should use the ``internal`` argument to access the expected value.

    Version Changed:
        6.0:
        Moved and renamed from rbtools.api.decorators.request_method_decorator.

    Args:
        f (callable):
            The method to wrap.

    Returns:
        callable:
        The wrapped method.
    """
    @wraps(f)
    def request_method(
        self: Resource,
        *args,
        **kwargs,
    ) -> RequestMethodResult:
        if kwargs.pop('internal', False):
            return f(self, *args, **kwargs)
        else:
            def method_wrapper(*args, **kwargs) -> HttpRequest:
                return f(self, *args, **kwargs)

            return self._transport.execute_request_method(method_wrapper,
                                                          *args, **kwargs)

    return request_method


_STUB_ATTR_NAME = '_rbtools_api_stub'


def api_stub(
    f: Callable[_P, _T],
) -> Callable[_P, _T]:
    """Mark a method as being an API stub.

    Version Added:
        6.0

    Args:
        f (callable):
            The stub method.

    Returns:
        callable:
        The stub method.
    """
    setattr(f, _STUB_ATTR_NAME, True)

    return f


def is_api_stub(
    f: Callable[..., Any],
) -> bool:
    """Return whether a given method is an API stub.

    Version Added:
        6.0

    Args:
        f (callable):
            The method to check.

    Returns:
        bool:
        ``True`` if the method was decorated with :py:func:`api_stub`.
        ``False``, otherwise.
    """
    return getattr(f, _STUB_ATTR_NAME, False)


def replace_api_stub(
    obj: Resource,
    attr: str,
    stub: Callable[..., Any],
    implementation: Callable[..., Any],
) -> None:
    """Replace an API stub with a real implementation.

    Version Added:
        6.0

    Args:
        obj (Resource):
            The resource object which owns the method.

        attr (str):
            The name of the method.

        stub (callable):
            The stub method.

        implementation (callable):
            The method implementation.
    """
    update_wrapper(implementation, stub)
    delattr(implementation, _STUB_ATTR_NAME)
    setattr(obj, attr, implementation)


def _preprocess_fields(
    fields: JSONDict,
) -> Iterator[tuple[str, str | bytes]]:
    """Pre-process request fields.

    Any ``extra_data_json`` (JSON Merge Patch) or ``extra_data_json_patch``
    (JSON Patch) fields will be serialized to JSON and stored.

    Any :samp:`extra_data__{key}` fields will be converted to
    :samp:`extra_data.{key}` fields, which will be handled by the Review Board
    API. These cannot store complex types.

    Version Changed:
        3.1:
        Added support for ``extra_data_json`` and ``extra_data_json_patch``.

    Args:
        fields (dict):
            A mapping of field names to field values.

    Yields:
        tuple:
        A 2-tuple of:

        1. The normalized field name to send in the request.
        2. The normalized value to send.
    """
    field_names = set(fields.keys())

    # Serialize the JSON Merge Patch or JSON Patch payloads first.
    for norm_field_name, field_name in (('extra_data:json',
                                         'extra_data_json'),
                                        ('extra_data:json-patch',
                                         'extra_data_json_patch')):
        if field_name in field_names:
            field_names.remove(field_name)

            yield (
                norm_field_name,
                json.dumps(fields[field_name],
                           sort_keys=True,
                           separators=(',', ':')),
            )

    for name in sorted(field_names):
        value = fields[name]

        if not isinstance(value, (str, bytes)):
            value = str(value)

        if name.startswith(_EXTRA_DATA_PREFIX):
            # It's technically not a problem to send both an extra_data.<key>
            # and a JSON Patch or Merge Patch in the same request, but in
            # the future we may want to warn about it, just to help guide
            # users toward a single implementation.
            key = name.removeprefix(_EXTRA_DATA_PREFIX)
            name = f'extra_data.{key}'

        yield name, value


def _create_resource_for_field(
    *,
    parent_resource: Resource,
    field_payload: JSONDict,
    mimetype: str | None,
    url: str,
    force_resource_type: (type[Resource] | None) = None,
) -> Resource:
    """Create a resource instance based on field data.

    This will construct a resource instance for the payload of a field,
    using the given mimetype to identify it. This is intended for use with
    expanded resources or items in lists.

    Version Changed:
        6.0:
        * Added the ``force_resource_type`` parameter.
        * Removed the ``item_mimetype`` parameter.
        * Made ``url`` parameter required.
        * Made arguments keyword-only.

    Args:
        parent_resource (Resource):
            The resource containing the field payload.

        field_payload (dict):
            The field payload to use as the new resource's payload.

        mimetype (str):
            The mimetype of the resource.

        url (str, optional):
            The URL of the resource, if one is available.

        force_resource_type (type, optional):
            The resource class to instantiate.

            Version Added:
                6.0
    """
    # We need to import this here to avoid circular imports.
    from rbtools.api.factory import create_resource

    return create_resource(transport=parent_resource._transport,
                           payload=field_payload,
                           url=url,
                           mime_type=mimetype,
                           guess_token=False,
                           force_resource_type=force_resource_type)


@request_method
def _create(
    resource: Resource,
    data: (dict[str, Any] | None) = None,
    query_args: (dict[str, QueryArgs] | None) = None,
    *args,
    **kwargs,
) -> HttpRequest:
    """Generate a POST request on a resource.

    Any ``extra_data_json`` (JSON Merge Patch) or ``extra_data_json_patch``
    (JSON Patch) fields will be serialized to JSON and stored.

    Any :samp:`extra_data__{key}` fields will be converted to
    :samp:`extra_data.{key}` fields, which will be handled by the Review Board
    API. These cannot store complex types.

    Version Changed:
        3.1:
        Added support for ``extra_data_json`` and ``extra_data_json_patch``.

    Args:
        resource (Resource):
            The resource instance owning this create method.

        data (dict, optional):
            Data to send in the POST request. This will be merged with
            ``**kwargs``.

        query_args (dict, optional):
            Optional query arguments for the URL.

        *args (tuple, unused):
            Unused positional arguments.

        **kwargs (dict):
            Keyword arguments representing additional fields to set in the
            request. This will be merged with ``data``.

    Returns:
        rbtools.api.request.HttpRequest:
        The resulting HTTP POST request for this create operation.

    Raises:
        rbtools.api.errors.APIError:
            The Review Board API returned an error.

        rbtools.api.errors.ServerInterfaceError:
            An error occurred while communicating with the server.
    """
    request = resource._make_httprequest(
        url=resource._links['create']['href'],
        method='POST',
        query_args=query_args)

    field_data = kwargs

    if data:
        field_data.update(data)

    for name, value in _preprocess_fields(field_data):
        request.add_field(name, value)

    return request


@request_method
def _delete(
    resource: Resource,
    *args,
    **kwargs: QueryArgs,
) -> HttpRequest:
    """Generate a DELETE request on a resource.

    Argrs:
        resource (Resource):
            The resource instance owning this method.

        *args (tuple, unused):
            Unused positional arguments.

        **kwargs (dict):
            Additional query arguments to include with the request.

    Returns:
        rbtools.api.request.HttpRequest:
        The HTTP request.

    Raises:
        rbtools.api.errors.APIError:
            The Review Board API returned an error.

        rbtools.api.errors.ServerInterfaceError:
            An error occurred while communicating with the server.
    """
    return resource._make_httprequest(
        url=resource._links['delete']['href'],
        method='DELETE',
        query_args=kwargs)


@request_method
def _get_self(
    resource: Resource,
    *args,
    **kwargs: QueryArgs,
) -> HttpRequest:
    """Generate a request for a resource's 'self' link.

    Args:
        resource (Resource):
            The resource instance owning this method.

        *args (tuple, unused):
            Unused positional arguments.

        **kwargs (dict):
            Additional query arguments to include with the request.

    Returns:
        rbtools.api.request.HttpRequest:
        The HTTP request.

    Raises:
        rbtools.api.errors.APIError:
            The Review Board API returned an error.

        rbtools.api.errors.ServerInterfaceError:
            An error occurred while communicating with the server.
    """
    return resource._make_httprequest(
        url=resource._links['self']['href'],
        query_args=kwargs)


@request_method
def _update(
    resource: Resource,
    data: (dict[str, Any] | None) = None,
    query_args: (dict[str, QueryArgs] | None) = None,
    *args,
    **kwargs,
) -> HttpRequest:
    """Generate a PUT request on a resource.

    Any ``extra_data_json`` (JSON Merge Patch) or ``extra_data_json_patch``
    (JSON Patch) fields will be serialized to JSON and stored.

    Any :samp:`extra_data__{key}` fields will be converted to
    :samp:`extra_data.{key}` fields, which will be handled by the Review Board
    API. These cannot store complex types.

    Version Changed:
        3.1:
        Added support for ``extra_data_json`` and ``extra_data_json_patch``.

    Args:
        resource (Resource):
            The resource instance owning this create method.

        data (dict, optional):
            Data to send in the PUT request. This will be merged with
            ``**kwargs``.

        query_args (dict, optional):
            Optional query arguments for the URL.

        *args (tuple, unused):
            Unused positional arguments.

        **kwargs (dict):
            Keyword arguments representing additional fields to set in the
            request. This will be merged with ``data``.

    Returns:
        rbtools.api.request.HttpRequest:
        The resulting HTTP PUT request for this update operation.

    Raises:
        rbtools.api.errors.APIError:
            The Review Board API returned an error.

        rbtools.api.errors.ServerInterfaceError:
            An error occurred while communicating with the server.
    """
    request = resource._make_httprequest(
        url=resource._links['update']['href'],
        method='PUT',
        query_args=query_args)

    field_data = kwargs

    if data:
        field_data.update(data)

    for name, value in _preprocess_fields(field_data):
        request.add_field(name, value)

    return request


# This dictionary is a mapping of special keys in a resources links,
# to a name and method used for generating a request for that link.
# This is used to special case the REST operation links. Any link
# included in this dictionary will be generated separately, and links
# with a None for the method will be ignored.
SPECIAL_LINKS: Mapping[
    str,
    tuple[str, Callable[..., RequestMethodResult] | None]
] = {
    'create': ('create', _create),
    'delete': ('delete', _delete),
    'next': ('get_next', None),
    'prev': ('get_prev', None),
    'self': ('get_self', _get_self),
    'update': ('update', _update),
}


class ResourceLink(TypedDict):
    """Type for a link within a payload.

    Version Added:
        6.0
    """

    #: The link URL.
    href: str

    #: The HTTP method to use for the link.
    method: str

    #: The MIME type of the object located at the link.
    mimetype: NotRequired[str]

    #: The user-visible title of the object located at the link.
    title: NotRequired[str]


#: Type for link data within the payload.
#:
#: Version Added:
#:     6.0
ResourceLinks = dict[str, ResourceLink]


class ExpandInfo(TypedDict):
    """Information on expanded resources.

    This corresponds to :py:class:`djblets.webapi.resources.base._ExpandInfo`.

    Version Added:
        6.0
    """

    #: The MIME type of an expanded item resource.
    item_mimetype: str

    #: The MIME type of an expanded list resource.
    list_mimetype: NotRequired[str]

    #: The URL to an expanded list resource, if any.
    list_url: NotRequired[str | None]


class BaseGetParams(TypedDict, total=False):
    """Base class for parameters for GET requests.

    This has the basic fields that are supported by all resource endpoints.

    Version Added:
        6.0
    """

    #: A comma-separated list of links to expand within the returned payload.
    expand: str

    #: A comma-separated list of fields to limit the payload to.
    only_fields: str

    #: A comma-separated list of links to limit the payload to.
    only_links: str


class BaseGetListParams(BaseGetParams, total=False):
    """Base class for parameters for GET requests on lists.

    Version Added:
        6.0
    """

    #: If specified, return only the counts.
    #:
    #: Making a request with this will return a :py:class:`CountResource`
    #: instead of a list.
    counts_only: bool

    #: The maximum number of results to return in the request.
    #:
    #: By default, this is 25. There is a hard limit of 200; if you need more
    #: than 200 results, you will need to make more than one request.
    max_results: int

    #: The 0-based index of the first result in the list.
    #:
    #: To page through results, the start index should be set to the previous
    #: start index plus the number of previous results.
    start: int


class Resource:
    """Defines common functionality for Item and List Resources.

    Resources are able to make requests to the Web API by returning an
    HttpRequest object. When an HttpRequest is returned from a method
    call, the transport layer will execute this request and return the
    result to the user.

    Methods for constructing requests to perform each of the supported
    REST operations will be generated automatically. These methods
    will have names corresponding to the operation (e.g. 'update()').
    An additional method for re-requesting the resource using the
    'self' link will be generated with the name 'get_self'. Each
    additional link will have a method generated which constructs a
    request for retrieving the linked resource.
    """

    #: Attributes which should be excluded when processing the payload.
    #:
    #: Version Changed:
    #:     6.0:
    #:     Changed the type from a list to a set.
    _excluded_attrs: ClassVar[set[str]] = set()

    #: Links which should be excluded when processing the payload.
    _excluded_links: ClassVar[set[str]] = set()

    #: A mapping for rewriting HTTP query parameters.
    #:
    #: The Review Board Web API uses hyphens for many parameters, but that
    #: doesn't work well when trying to pass those via Python method kwargs.
    #:
    #: Version Added:
    #:     6.0
    _httprequest_params_name_map: ClassVar[Mapping[str, str]] = {
        'only_fields': 'only-fields',
        'only_links': 'only-links',
    }

    ######################
    # Instance variables #
    ######################

    #: Information about expanded fields in the payload.
    _expanded_info: dict[str, ExpandInfo]

    #: The links for the resource.
    _links: ResourceLinks

    #: The full resource payload.
    _payload: JSONDict

    #: The key within the request payload for the resource data.
    #:
    #: If this is ``None``, the payload contains the resource data directly.
    _token: str | None

    #: The API transport.
    _transport: Transport

    #: The resource URL.
    _url: str

    def __init__(
        self,
        transport: Transport,
        payload: JSONDict,
        url: str,
        token: (str | None) = None,
        **kwargs,
    ) -> None:
        """Initialize the resource.

        Args:
            transport (rbtools.api.transport.Transport):
                The API transport.

            payload (dict):
                The request payload.

            url (str):
                The URL for the resource.

            token (str, optional):
                The key within the request payload for the resource data.

            **kwargs (dict, unused):
                Unused keyword arguments.
        """
        self._url = url
        self._transport = transport
        self._token = token
        self._payload = payload

        # Determine where the links live in the payload. This
        # can either be at the root, or inside the resources
        # token.
        if LINKS_TOK in self._payload:
            self._links = cast(ResourceLinks, self._payload[LINKS_TOK])
        elif (token and isinstance(self._payload[token], dict) and
              (token_payload := self._payload[token]) and
              isinstance(token_payload, dict) and
              LINKS_TOK in token_payload):
            self._links = cast(ResourceLinks, token_payload[LINKS_TOK])
        else:
            self._payload[LINKS_TOK] = {}
            self._links = {}

        # If we've expanded any fields, we'll try to convert the expanded
        # payloads into resources. We can only do this if talking to
        # Review Board 4.0+.
        if EXPANDED_TOKEN in self._payload:
            self._expanded_info = cast(
                dict[str, ExpandInfo],
                self._payload[EXPANDED_TOKEN])
        elif (token and
              (token_payload := self._payload[token]) and
              isinstance(token_payload, dict) and
              EXPANDED_TOKEN in token_payload):
            self._expanded_info = cast(
                dict[str, ExpandInfo],
                token_payload[EXPANDED_TOKEN])
        else:
            self._expanded_info = {}

        members = set(dir(self))

        # Add a method for each supported REST operation, and
        # for retrieving 'self'.
        for link, method in SPECIAL_LINKS.items():
            if link in self._links and method[1]:
                attr_name = method[0]

                def special_method(
                    resource: Self = self,
                    meth: Callable[..., RequestMethodResult] = method[1],
                    **kwargs,
                ) -> RequestMethodResult:
                    return meth(resource, **kwargs)

                if attr_name not in members:
                    # This log message is useful for adding new stubs.
                    logger.debug('%s is missing API stub for %s',
                                 self.__class__.__name__, attr_name)
                    setattr(self, attr_name, special_method)
                elif is_api_stub(stub := getattr(self, attr_name)):
                    replace_api_stub(self, attr_name, stub, special_method)

        # Generate request methods for any additional links the resource has.
        excluded_links = self._excluded_links

        for link, body in self._links.items():
            if link not in SPECIAL_LINKS and link not in excluded_links:
                # Some resources have hyphens in the links.
                attr_name = f'get_{link.replace("-", "_")}'
                url = body['href']

                def link_method(
                    resource: Self = self,
                    url: str = url,
                    **kwargs,
                ) -> RequestMethodResult:
                    return resource._get_url(url, **kwargs)

                if attr_name not in members:
                    # This log message is useful for adding new stubs.
                    logger.debug('%s is missing API stub for %s (%s)',
                                 self.__class__.__name__, attr_name, url)
                    setattr(self, attr_name, link_method)
                elif is_api_stub(stub := getattr(self, attr_name)):
                    replace_api_stub(self, attr_name, stub, link_method)

    def _make_httprequest(
        self,
        *,
        url: str,
        method: str = 'GET',
        query_args: (Mapping[Any, Any] | None) = None,
        headers: (Mapping[str, str] | None) = None,
    ) -> HttpRequest:
        """Create an HTTP request.

        This will handle renaming any query arguments.

        Version Added:
            6.0

        Args:
            url (str):
                The URL to request.

            method (str, optional):
                The HTTP method to send to the server.

            query_args (dict, optional):
                Any query arguments to add to the URL.

            headers (dict, optional):
                Any HTTP headers to provide in the request.

        Returns:
            rbtools.api.request.HttpRequest:
            The HTTP request object.
        """
        if query_args:
            def rewrite_query_arg(key: str) -> str:
                return self._httprequest_params_name_map.get(key, key)

            query_args = {
                rewrite_query_arg(key): value
                for key, value in query_args.items()
            }

        return HttpRequest(url, method, query_args, headers)

    def _wrap_field(
        self,
        field_payload: Any,
        field_name: (str | None) = None,
        field_url: (str | None) = None,
        field_mimetype: (str | None) = None,
        list_item_mimetype: (str | None) = None,
        force_resource: bool = False,
        *,
        force_resource_type: (type[Resource] | None) = None,
    ) -> Any:
        """Wrap the value of a field in a resource or field object.

        This determines a suitable wrapper for a field, turning it into
        a resource or a wrapper with utility methods that can be used to
        interact with the field or perform additional queries.

        Version Changed:
            6.0:
            Added the ``force_resource_type`` argument.

        Args:
            field_payload (object):
                The payload of the field. The type of value determines the
                way in which this is wrapped.

            field_name (str, optional):
                The name of the field being wrapped, if known.

                Version Added:
                    3.1

            field_url (str, optional):
                The URL representing the payload in the field, if one is
                available. If not provided, one may be computed, depending
                on the type and contents of the field.

            field_mimetype (str, optional):
                The mimetype used to represent the field. If provided, this
                may result in the wrapper being a :py:class:`Resource`
                subclass.

            list_item_mimetype (str, optional):
                The mimetype used or any items within the field, if the
                field is a list.

            force_resource (bool, optional):
                Force the return of a resource, even a generic one, instead
                of a field wrapper.

            force_resource_type (type, optional):
                The resource class to instantiate.

                Version Added:
                    6.0

        Returns:
            object:
            A wrapper, or the field payload. This may be one of:

            1. A subclass of :py:class:`Resource`.
            2. A field wrapper (:py:class:`ResourceDictField`,
               :py:class:`ResourceListField`,
               :py:class:`ResourceLinkField`, or
               :py:class:`ResourceExtraDataField`).
            3. The field payload itself, if no wrapper is needed.
        """
        if isinstance(field_payload, dict):
            if (force_resource or
                (field_mimetype and
                 rem_mime_format(field_mimetype) in RESOURCE_MAP)):
                # We have a resource backing this mimetype. Try to create
                # an instance of the resource for this payload.
                if not field_url:
                    try:
                        field_url = \
                            cast(str, field_payload['links']['self']['href'])
                    except KeyError:
                        field_url = ''

                return _create_resource_for_field(
                    parent_resource=self,
                    field_payload=field_payload,
                    mimetype=field_mimetype,
                    url=field_url,
                    force_resource_type=force_resource_type)
            elif field_name == 'extra_data':
                # If this is an extra_data field, we'll return a special
                # ExtraDataField.
                return ResourceExtraDataField(resource=self,
                                              fields=field_payload)
            elif ('href' in field_payload and
                  not set(field_payload.keys()) - LINK_KEYS):
                # If the payload consists solely of link-supported keys,
                # then we'll return a special ResourceLinkField.
                return ResourceLinkField(resource=self,
                                         field_payload=field_payload)
            else:
                # Anything else is treated as a standard dictionary, which
                # will be wrapped.
                return ResourceDictField(resource=self,
                                         fields=field_payload)
        elif isinstance(field_payload, list):
            return ResourceListField(self, field_payload,
                                     item_mimetype=list_item_mimetype)
        else:
            return field_payload

    @property
    def links(self) -> ResourceDictField:
        """The resource's links.

        This is a special property which allows direct access to the links
        dictionary for a resource. Unlike other properties which come from the
        resource fields, this one is only accessible as a property, and not
        using array syntax.
        """
        return ResourceDictField(self, self._links)

    @request_method
    def _get_url(
        self,
        url: str,
        **kwargs: QueryArgs,
    ) -> HttpRequest:
        """Make a GET request to a given URL.

        Args:
            url (str):
                The URL to fetch.

            **kwargs (dict):
                Additional query arguments to include with the request.

        Returns:
            rbtools.api.request.HttpRequest:
            The HTTP request.
        """
        return self._make_httprequest(url=url, query_args=kwargs)

    @property
    def rsp(self) -> JSONDict:
        """Return the response payload used to create the resource.

        Returns:
            dict:
            The response payload.
        """
        return self._payload

    @api_stub
    def get_self(
        self,
        *args,
        **kwargs: Unpack[BaseGetParams],
    ) -> Self:
        """Get the resource's 'self' link.

        Args:
            *args (tuple, unused):
                Unused positional arguments.

            **kwargs (dict):
                Query arguments to include with the request.

        Returns:
            Resource:
            The newly-fetched resource instance.

        Raises:
            rbtools.api.errors.APIError:
                The Review Board API returned an error.

            rbtools.api.errors.ServerInterfaceError:
                An error occurred while communicating with the server.
        """
        raise NotImplementedError


_TResource = TypeVar('_TResource', bound=Resource)
_TResourceClass = TypeVar('_TResourceClass', bound=type[Resource])


def resource_mimetype(
    mimetype: str,
) -> Callable[[_TResourceClass], _TResourceClass]:
    """Set the mimetype for the decorated class in the resource map.

    Args:
        mimetype (str):
            The MIME type for the resource.

    Returns:
        callable:
        A decorator to apply to a resource class.
    """
    def wrapper(cls: _TResourceClass) -> _TResourceClass:
        RESOURCE_MAP[mimetype] = cls
        return cls

    return wrapper


#: The resulting type of a resource method.
#:
#: Version Added:
#:     6.0
RequestMethodResult = HttpRequest | Resource | None


class request_method_returns(Generic[_TResource]):
    """Decorator to mark request methods with a specific return type.

    When the return type of a method is known to be a specific type, this
    decorator can be used in place of @request_method, and will rewrite the
    annotation of the return value of the decorated method.

    Version Added:
        6.0
    """

    def __call__(
        self,
        f: Callable[_P, HttpRequest],
    ) -> Callable[_P, _TResource]:
        """Decorate the method.

        Args:
            f (callable):
                The method to decorate.
        """
        @request_method
        def wrapper(
            self: Resource,
            *args,
            **kwargs,
        ) -> HttpRequest:
            return f(self, *args, **kwargs)

        update_wrapper(wrapper, f)

        return cast(Callable[_P, _TResource], wrapper)


class ResourceDictField(MutableMapping[str, Any]):
    """Wrapper for dictionaries returned from a resource.

    Items fetched from this dictionary may be wrapped as a resource or
    resource field container.

    Changes cannot be made to resource dictionaries. Instead, changes must be
    made using :py:meth:`Resource.update` calls.

    Version Changed:
        3.1:
        This class now operates like a standard dictionary, but blocks any
        changes (which were misleading and could not be used to save state
        in any prior version).
    """

    ######################
    # Instance variables #
    ######################

    #: The wrapped fields dictionary.
    _fields: dict[str, Any]

    #: The resource which owns this field.
    _resource: Resource

    def __init__(
        self,
        resource: Resource,
        fields: dict[str, Any],
    ) -> None:
        """Initialize the field.

        Args:
            resource (Resource):
                The parent resource that owns this field.

            fields (dict):
                The dictionary contents from the payload.
        """
        super().__init__()

        self._resource = resource
        self._fields = fields

    def __getattr__(
        self,
        name: str,
    ) -> Any:
        """Return the value of a key from the field as an attribute reference.

        The resulting value will be wrapped as a resource or resource field
        if appropriate.

        Args:
            name (str):
                The name of the key.

        Returns:
            object:
            The value of the field.

        Raises:
            AttributeError:
                The provided key name was not found in the dictionary.
        """
        try:
            return self._wrap_field(name)
        except KeyError:
            raise AttributeError(
                f'This dictionary resource for '
                f'{self._resource.__class__.__name__} does not have an '
                f'attribute "{name}".')

    def __getitem__(
        self,
        name: str,
    ) -> Any:
        """Return the value of a key from the field as an item lookup.

        The resulting value will be wrapped as a resource or resource field
        if appropriate.

        Args:
            name (str):
                The name of the key.

        Returns:
            object:
            The value of the field.

        Raises:
            KeyError:
                The provided key name was not found in the dictionary.
        """
        try:
            return self._wrap_field(name)
        except KeyError:
            raise KeyError(
                f'This dictionary resource for '
                f'{self._resource.__class__.__name__} does not have a key '
                f'"{name}".')

    def __delitem__(
        self,
        name: str,
    ) -> None:
        """Delete an item from the dictionary.

        This will raise an exception stating that changes are not allowed
        and offering an alternative.

        Args:
            name (str, unused):
                The name of the key to delete.

        Raises:
            AttributeError:
                An error stating that changes are not allowed.
        """
        self._raise_immutable()

    def __setitem__(
        self,
        name: str,
        value: Any,
    ) -> None:
        """Set an item in the dictionary.

        This will raise an exception stating that changes are not allowed
        and offering an alternative.

        Args:
            name (str, unused):
                The name of the key to set.

            value (object, unused):
                The value to set.

        Raises:
            AttributeError:
                An error stating that changes are not allowed.
        """
        self._raise_immutable()

    def __len__(self) -> int:
        """Return the number of items in the dictionary.

        Returns:
            int:
            The number of items.
        """
        return len(self._fields)

    def __iter__(self) -> Iterator[Any]:
        """Iterate through the dictionary.

        Yields:
            object:
            Each item in the dictionary.
        """
        yield from self._fields.keys()

    def __repr__(self) -> str:
        """Return a string representation of the dictionary field.

        Returns:
            str:
            The string representation.
        """
        return (f'{self.__class__.__name__}(resource={self._resource!r}, '
                f'fields={self._fields!r})')

    def fields(self) -> Iterator[str]:
        """Iterate through all fields in the dictionary.

        This will yield each field name in the dictionary. This is the same
        as calling :py:meth:`keys` or simply ``for field in dict_field``.

        Yields:
            str:
            Each field in this dictionary.
        """
        yield from self

    def _wrap_field(
        self,
        field_name: str,
    ) -> Any:
        """Conditionally return a wrapped version of a field's value.

        This will wrap content according to the resource's wrapping logic.

        Args:
            field_name (str):
                The name of the field to wrap.

        Returns:
            object:
            The wrapped object or field value.

        Raises:
            KeyError:
                The field could not be found in this dictionary.
        """
        # This may raise an exception, which will be handled by the caller.
        return self._resource._wrap_field(self._fields[field_name],
                                          field_name=field_name)

    def _raise_immutable(self) -> NoReturn:
        """Raise an exception stating that the dictionary is immutable.

        Version Added:
            3.1

        Raises:
            AttributeError:
                An error stating that changes are not allowed.
        """
        raise AttributeError(
            'Attributes cannot be modified directly on this dictionary. To '
            'change values, issue a .update(attr=value, ...) call on the '
            'parent resource.')


class ResourceLinkField(ResourceDictField, Generic[_TResource]):
    """Wrapper for link dictionaries returned from a resource.

    In order to support operations on links found outside of a
    resource's links dictionary, detected links are wrapped with this
    class.

    A links fields (href, method, and title) are accessed as
    attributes, and link operations are supported through method
    calls. Currently the only supported method is "GET", which can be
    invoked using the 'get' method.
    """

    ######################
    # Instance variables #
    ######################

    #: The API transport.
    _transport: Transport

    def __init__(
        self,
        resource: Resource,
        field_payload: JSONDict,
    ) -> None:
        """Initialize the resource.

        Args:
            resource (Resource):
                The resource which owns this field.

            field_payload (dict):
                The field content.
        """
        super().__init__(resource, field_payload)
        self._transport = resource._transport

    @request_method_returns[_TResource]()
    def get(
        self,
        **query_args: QueryArgs,
    ) -> HttpRequest:
        """Fetch the link.

        Args:
            **query_args (dict):
                Query arguments to include with the request.

        Returns:
            rbtools.api.request.HttpRequest:
            The HTTP request.
        """
        return self._resource._make_httprequest(
            url=self._fields['href'], query_args=query_args)


class ResourceExtraDataField(ResourceDictField):
    """Wrapper for extra_data fields on resources.

    Version Added:
        3.1
    """

    def copy(self) -> dict[str, Any]:
        """Return a copy of the dictionary's fields.

        A copy of the original ``extra_data`` content will be returned,
        without any field wrapping.

        Returns:
            dict:
            The copy of the dictionary.
        """
        return copy.deepcopy(self._fields)

    def _wrap_field(
        self,
        field_name: str,
    ) -> JSONValue | ResourceExtraDataField:
        """Conditionally return a wrapped version of a field's value.

        This will wrap dictionaries in another
        :py:class:`ResourceExtraDataField`, and otherwise leave everything
        else unwrapped (preventing list-like or links-like payloads from
        being wrapped in their respective field types).

        Args:
            field_name (str):
                The name of the field to wrap.

        Returns:
            object:
            The wrapped object or field value.

        Raises:
            KeyError:
                The field could not be found in this dictionary.
        """
        # This may raise an exception, which will be handled by the caller.
        value = self._fields[field_name]

        if isinstance(value, dict):
            return ResourceExtraDataField(resource=self._resource,
                                          fields=value)

        # Leave everything else unwrapped.
        return value

    def _raise_immutable(self) -> NoReturn:
        """Raise an exception stating that the dictionary is immutable.

        Raises:
            AttributeError:
                An error stating that changes are not allowed.
        """
        raise AttributeError(
            f'extra_data attributes cannot be modified directly on this '
            f'dictionary. To make a mutable copy of this and all its '
            f'contents, call .copy(). To set or change extra_data state, '
            f'issue a .update(extra_data_json={{...}}) for a JSON Merge '
            f'Patch request or .update(extra_data_json_patch=[...]) for a '
            f'JSON Patch request on the parent resource. See '
            f'{_EXTRA_DATA_DOCS_URL} for the format for these operations.')


_TListValue = TypeVar('_TListValue')


class ResourceListField(Generic[_TListValue], list[_TListValue]):
    """Wrapper for lists returned from a resource.

    Acts as a normal list, but wraps any returned items.
    """

    ######################
    # Instance variables #
    ######################

    #: The resource which owns the field.
    _resource: Resource

    #: The MIME type of items in the list.
    _item_mimetype: str | None

    def __init__(
        self,
        resource: Resource,
        list_field: list[_TListValue],
        item_mimetype: (str | None) = None,
    ) -> None:
        """Initialize the field.

        Args:
            resource (Resource):
                The resource which owns this field.

            list_field (list):
                The list contents.

            item_mimetype (str, optional):
                The mimetype of the list items.
        """
        super().__init__(list_field)

        self._resource = resource
        self._item_mimetype = item_mimetype

    def __getitem__(
        self,
        key: int,
    ) -> _TListValue:
        """Return the item at the given index.

        Args:
            key (int):
                The index to fetch.

        Returns:
            object:
            The item at the given index.
        """
        item = super().__getitem__(key)

        return self._resource._wrap_field(item,
                                          field_mimetype=self._item_mimetype)

    def __iter__(self) -> Iterator[_TListValue]:
        """Iterate through the list.

        Yields:
            object:
            Each item in the list.
        """
        for item in super().__iter__():
            yield self._resource._wrap_field(
                item,
                field_mimetype=self._item_mimetype)

    def __repr__(self) -> str:
        """Return a string representation of the field.

        Returns:
            str:
            A string representation of the field.
        """
        return (f'{self.__class__.__name__}(resource={self._resource}, '
                f'item_mimetype={self._item_mimetype}, '
                f'list_field={super().__repr__()})')


class ItemResource(Resource):
    """The base class for Item Resources.

    Any resource specific base classes for Item Resources should
    inherit from this class. If a resource specific base class does
    not exist for an Item Resource payload, this class will be used to
    create the resource.

    The body of the resource is copied into the fields dictionary. The
    Transport is responsible for providing access to this data,
    preferably as attributes for the wrapping class.
    """

    ######################
    # Instance variables #
    ######################

    #: The fields payload data.
    _fields: JSONDict

    def __init__(
        self,
        transport: Transport,
        payload: JSONDict,
        url: str,
        token: (str | None) = None,
        **kwargs,
    ) -> None:
        """Initialize the resource.

        Args:
            transport (rbtools.api.transport.Transport):
                The API transport.

            payload (dict):
                The resource payload.

            url (str):
                The resource URL.

            token (str, optional):
                The key within the request payload for the resource data.

            **kwargs (dict, unused):
                Unused keyword arguments.
        """
        super().__init__(transport, payload, url, token=token, **kwargs)
        self._fields = {}

        # Determine the body of the resource's data.
        if token is not None:
            data = cast(JSONDict, self._payload[token])
        else:
            data = self._payload

        excluded_attrs = self._excluded_attrs | _EXCLUDE_ATTRS

        for name, value in data.items():
            if name not in excluded_attrs:
                self._fields[name] = value

    def __getattr__(
        self,
        name: str,
    ) -> Any:
        """Return the value for an attribute on the resource.

        If the attribute represents an expanded resource, and there's
        information available on the expansion (available in Review Board
        4.0+), then a resource instance will be returned.

        If the attribute otherwise represents a dictionary, list, or a link,
        a wrapper may be returned.

        Args:
            name (str):
                The name of the attribute.

        Returns:
            object:
            The attribute value, or a wrapper or resource representing that
            value.

        Raises:
            AttributeError:
                A field with the given attribute name was not found.
        """
        try:
            field_payload = self._fields[name]
        except KeyError:
            raise AttributeError(
                f'This {self.__class__.__name__} does not have an attribute '
                f'"{name}".')

        expand_info = self._expanded_info.get(name, {})

        if isinstance(field_payload, dict):
            value = self._wrap_field(
                field_name=name,
                field_payload=field_payload,
                field_mimetype=expand_info.get('item_mimetype'))
        elif isinstance(field_payload, list):
            value = self._wrap_field(
                field_name=name,
                field_payload=field_payload,
                field_url=expand_info.get('list_url'),
                field_mimetype=expand_info.get('list_mimetype'),
                list_item_mimetype=expand_info.get('item_mimetype'))
        else:
            value = self._wrap_field(field_payload,
                                     field_name=name)

        return value

    def __getitem__(
        self,
        key: str,
    ) -> Any:
        """Return the value for an attribute on the resource.

        If the attribute represents an expanded resource, and there's
        information available on the expansion (available in Review Board
        4.0+), then a resource instance will be returned.

        If the attribute otherwise represents a dictionary, list, or a link,
        a wrapper may be returned.

        Args:
            key (str):
                The name of the attribute.

        Returns:
            object:
            The attribute value, or a wrapper or resource representing that
            value.

        Raises:
            KeyError:
                A field with the given attribute name was not found.
        """
        try:
            return self.__getattr__(key)
        except AttributeError:
            raise KeyError

    def __contains__(
        self,
        key: str,
    ) -> bool:
        """Return whether the resource has a field with the given name.

        Args:
            key (str):
                The name of the field.

        Returns:
            bool:
            Whether a field with the given name exists.
        """
        return key in self._fields

    def iterfields(self) -> Iterator[str]:
        """Iterate through all field names in the resource.

        Yields:
            str:
            The name of each field name.
        """
        yield from self._fields

    def iteritems(self) -> Iterator[tuple[str, Any]]:
        """Iterate through all field/value pairs in the resource.

        Yields:
            tuple:
            A tuple in ``(field_name, value)`` form.
        """
        for key in self.iterfields():
            yield key, self.__getattr__(key)

    def __repr__(self) -> str:
        """Return a string representation of the resource.

        Returns:
            str:
            A string representation of the resource.
        """
        return (f'{self.__class__.__name__}(transport={self._transport}, '
                f'payload={self._payload}, url={self._url}, '
                f'token={self._token})')

    @api_stub
    def delete(
        self,
        *args,
        **kwargs: QueryArgs,
    ) -> None:
        """Delete the resource.

        Args:
            *args (tuple, unused):
                Unused positional arguments.

            **kwargs (dict):
                Query arguments to include with the request.

        Raises:
            rbtools.api.errors.APIError:
                The Review Board API returned an error.

            rbtools.api.errors.ServerInterfaceError:
                An error occurred while communicating with the server.
        """
        raise NotImplementedError

    @overload
    def update(
        self,
        data: (Mapping[str, Any] | None) = None,
        query_args: (Mapping[str, QueryArgs] | None) = None,
        *args,
        internal: Literal[False] = False,
        **kwargs,
    ) -> Self:
        ...

    @overload
    def update(
        self,
        data: (Mapping[str, Any] | None) = None,
        query_args: (Mapping[str, QueryArgs] | None) = None,
        *args,
        internal: Literal[True],
        **kwargs,
    ) -> HttpRequest:
        ...

    @api_stub
    def update(
        self,
        data: (Mapping[str, Any] | None) = None,
        query_args: (Mapping[str, QueryArgs] | None) = None,
        *args,
        **kwargs,
    ) -> Self | HttpRequest:
        """Update the resource.

        Any ``extra_data_json`` (JSON Merge Patch) or ``extra_data_json_patch``
        (JSON Patch) fields will be serialized to JSON and stored.

        Any :samp:`extra_data__{key}` fields will be converted to
        :samp:`extra_data.{key}` fields, which will be handled by the Review
        Board API. These cannot store complex types.

        Args:
            resource (Resource):
                The resource instance owning this create method.

            data (dict, optional):
                Data to send in the PUT request. This will be merged with
                ``**kwargs``.

            query_args (dict, optional):
                Optional query arguments for the URL.

            *args (tuple, unused):
                Unused positional arguments.

            **kwargs (dict):
                Keyword arguments representing additional fields to set in the
                request. This will be merged with ``data``.

        Returns:
            ItemResource:
            The updated resource instance.

        Raises:
            rbtools.api.errors.APIError:
                The Review Board API returned an error.

            rbtools.api.errors.ServerInterfaceError:
                An error occurred while communicating with the server.
        """
        raise NotImplementedError


class CountResource(ItemResource):
    """Resource returned by a query with 'counts-only' true.

    When a resource is requested using 'counts-only', the payload will
    not contain the regular fields for the resource. In order to
    special case all payloads of this form, this class is used for
    resource construction.
    """

    def __init__(
        self,
        transport: Transport,
        payload: JSONDict,
        url: str,
        **kwargs,
    ) -> None:
        """Initialize the resource.

        Args:
            transport (rbtools.api.transport.Transport):
                The API transport.

            payload (dict):
                The response payload.

            url (str):
                The URL for the resource.

            **kwargs (dict, unused):
                Unused keyword arguments.
        """
        super().__init__(transport, payload, url, token=None)

    @request_method
    def get_self(
        self,
        **kwargs: QueryArgs,
    ) -> HttpRequest:
        """Generate an GET request for the resource list.

        This will return an HttpRequest to retrieve the list resource
        which this resource is a count for. Any query arguments used
        in the request for the count will still be present, only the
        'counts-only' argument will be removed

        Args:
            **kwargs (dict):
                Query arguments to include with the request.
        """
        # TODO: Fix this. It is generating a new request for a URL with
        # 'counts-only' set to False, but RB treats the  argument being set
        # to any value as true.
        kwargs.update({'counts_only': False})

        return self._make_httprequest(url=self._url, query_args=kwargs)


TItemResource = TypeVar('TItemResource', bound=ItemResource)


class ListResource(Generic[TItemResource], Resource):
    """The base class for List Resources.

    Any resource specific base classes for List Resources should
    inherit from this class. If a resource specific base class does
    not exist for a List Resource payload, this class will be used to
    create the resource.

    Instances of this class will act as a sequence, providing access
    to the payload for each Item resource in the list. Iteration is
    over the page of item resources returned by a single request, and
    not the entire list of resources. To iterate over all item
    resources 'get_next()' or 'get_prev()' should be used to grab
    additional pages of items.
    """

    _httprequest_params_name_map: ClassVar[Mapping[str, str]] = {
        'counts_only': 'counts-only',
        'max_results': 'max-results',
        **Resource._httprequest_params_name_map,
    }

    #: A resource type to force for individual items.
    #:
    #: For most resources, this is unnecessary because MIME type matching will
    #: instantiate the correct item resource subclass. In cases where items do
    #: not have their own endpoint and therefore have no MIME type, setting
    #: this will ensure that the returned items are the correct item type.
    _item_resource_type: ClassVar[type[Resource] | None] = None

    ######################
    # Instance variables #
    ######################

    #: The number of items in the current page.
    #:
    #: Type:
    #:     int
    num_items: int

    #: The total number of results in the list across all pages.
    #:
    #: This is commonly set for most list resources, but is not always
    #: guaranteed to be available. Callers should check to make sure this is
    #: not ``None``.
    #:
    #: Type:
    #:     int
    total_results: int | None

    #: The raw items in the list payload.
    _item_list: list[JSONValue]

    #: The MIME type of items in the list.
    _item_mime_type: str | None

    def __init__(
        self,
        transport: Transport,
        payload: JSONDict,
        url: str,
        token: (str | None) = None,
        item_mime_type: (str | None) = None,
        **kwargs,
    ) -> None:
        """Initialize the resource.

        Args:
            transport (rbtools.api.transport.Transport):
                The API transport.

            payload (dict or list):
                The payload data.

            url (str):
                The URL for the resource.

            token (str, optional):
                The key within the request payload for the resource data.

            item_mime_type (str, optional):
                The mimetype of the items within the list.

            **kwargs (dict):
                Keyword arguments to pass through to the base class.
        """
        super().__init__(transport, payload, url, token=token, **kwargs)
        self._item_mime_type = item_mime_type

        # The token must always be present for list resources.
        assert token is not None

        self._item_list = cast(list[JSONValue], payload[token])

        self.num_items = len(self._item_list)
        self.total_results = cast(int, payload.get('total_results'))

    def __len__(self) -> int:
        """Return the length of the list.

        Returns:
            int:
            The number of items in the list.
        """
        return self.num_items

    def __bool__(self) -> bool:
        """Return whether the list is truthy.

        Returns:
            bool:
            ``True``, always.
        """
        return True

    def __getitem__(
        self,
        index: int,
    ) -> TItemResource:
        """Return the item at the specified index.

        Args:
            index (int):
                The index of the item to retrieve.

        Returns:
            object:
            The item at the specified index.

        Raises:
            IndexError:
                The index is out of range.
        """
        return self._wrap_field(self._item_list[index],
                                field_mimetype=self._item_mime_type,
                                force_resource=True,
                                force_resource_type=self._item_resource_type)

    def __iter__(self) -> Iterator[TItemResource]:
        """Iterate through the items.

        Yields:
            TItemResource:
            Each item in the list.
        """
        for i in range(self.num_items):
            yield self[i]

    @request_method
    def get_next(
        self,
        **kwargs: Unpack[BaseGetListParams],
    ) -> HttpRequest:
        """Return the next page of results.

        Args:
            **kwargs (dict):
                Query arguments to include with the request.

        Returns:
            rbtools.api.request.HttpRequest):
            The HTTP request.

        Raises:
            StopIteration:
                There are no more pages of results.
        """
        if 'next' not in self._links:
            raise StopIteration()

        return self._make_httprequest(url=self._links['next']['href'],
                                      query_args=kwargs)

    @request_method
    def get_prev(
        self,
        **kwargs: Unpack[BaseGetListParams],
    ) -> HttpRequest:
        """Return the previous page of results.

        Args:
            **kwargs (dict):
                Query arguments to include with the request.

        Returns:
            rbtools.api.request.HttpRequest):
            The HTTP request.

        Raises:
            StopIteration:
                There are no previous pages of results.
        """
        if 'prev' not in self._links:
            raise StopIteration()

        return self._make_httprequest(url=self._links['prev']['href'],
                                      query_args=kwargs)

    @request_method
    def get_item(
        self,
        pk: int,
        **kwargs: Unpack[BaseGetParams],
    ) -> HttpRequest:
        """Retrieve the item resource with the corresponding primary key.

        Args:
            pk (int):
                The primary key of the item to fetch.

            **kwargs (dict):
                Query arguments to include with the request.

        Returns:
            rbtools.api.request.HttpRequest:
            The HTTP request.
        """
        return self._make_httprequest(url=urljoin(self._url, f'{pk}/'),
                                      query_args=kwargs)

    @property
    def all_pages(self) -> Iterator[Self]:
        """Yield all pages of item resources.

        Each page of resources is itself an instance of the same
        ``ListResource`` class.
        """
        page = self

        while True:
            yield page

            try:
                page = cast(Self, page.get_next())
            except StopIteration:
                break

    @property
    def all_items(self) -> Iterator[TItemResource]:
        """Yield all item resources in all pages of this resource.

        Yields:
            TItemResource:
            All items in the list.
        """
        for page in self.all_pages:
            yield from page

    def __repr__(self) -> str:
        """Return a string representation of the resource.

        """
        return (f'{self.__class__.__name__}(transport={self._transport}, '
                f'payload={self._payload}, url={self._url}, '
                f'token={self._token}, item_mime_type={self._item_mime_type})')

    @overload
    def create(
        self,
        data: (dict[str, Any] | None) = None,
        query_args: (dict[str, QueryArgs] | None) = None,
        *args,
        internal: Literal[False] = False,
        **kwargs,
    ) -> TItemResource:
        ...

    @overload
    def create(
        self,
        data: (dict[str, Any] | None) = None,
        query_args: (dict[str, QueryArgs] | None) = None,
        *args,
        internal: Literal[True],
        **kwargs,
    ) -> HttpRequest:
        ...

    @api_stub
    def create(
        self,
        data: (dict[str, Any] | None) = None,
        query_args: (dict[str, QueryArgs] | None) = None,
        *args,
        **kwargs,
    ) -> TItemResource | HttpRequest:
        """Create an item resource.

        Any ``extra_data_json`` (JSON Merge Patch) or ``extra_data_json_patch``
        (JSON Patch) fields will be serialized to JSON and stored.

        Any :samp:`extra_data__{key}` fields will be converted to
        :samp:`extra_data.{key}` fields, which will be handled by the Review
        Board API. These cannot store complex types.

        Args:
            resource (Resource):
                The resource instance owning this create method.

            data (dict, optional):
                Data to send in the POST request. This will be merged with
                ``**kwargs``.

            query_args (dict, optional):
                Optional query arguments for the URL.

            *args (tuple, unused):
                Unused positional arguments.

            **kwargs (dict):
                Keyword arguments representing additional fields to set in the
                request. This will be merged with ``data``.

        Returns:
            ItemResource:
            The newly-created item resource.

        Raises:
            rbtools.api.errors.APIError:
                The Review Board API returned an error.

            rbtools.api.errors.ServerInterfaceError:
                An error occurred while communicating with the server.
        """
        raise NotImplementedError


#: Constants for text type fields.
#:
#: Version Added:
#:     6.0
TextType = Literal['plain', 'markdown', 'html']
