"""Resource definitions for API tokens.

Version Added:
    6.0
"""

from __future__ import annotations

from typing import TYPE_CHECKING

from rbtools.api.resource.base import (
    ItemResource,
    ListResource,
    resource_mimetype,
)

if TYPE_CHECKING:
    from typelets.json import JSONDict

    from rbtools.api.resource.base import ResourceExtraDataField


@resource_mimetype('application/vnd.reviewboard.org.api-token')
class APITokenItemResource(ItemResource):
    """Item resource for API tokens.

    This corresponds to Review Board's :ref:`rb:webapi2.0-api-token-resource`.

    Version Added:
        6.0
    """

    ######################
    # Instance variables #
    ######################

    #: Whether the token is deprecated.
    #:
    #: This will be ``True`` if the token was generated by a deprecated token
    #: generator.
    deprecated: bool

    #: Whether the token is expired.
    expired: bool

    #: An optional field for the date and time that the token will expire.
    #:
    #: This is in ISO-8601 format. The token will be invalid and unusable for
    #: authentication after this point.
    expires: str | None

    #: Extra data as part of the token.
    extra_data: ResourceExtraDataField

    #: The numeric ID of the API token.
    id: int

    #: The date and time at which the token became invalid, in ISO-8601 format.
    invalid_date: str | None

    #: A message explaining why the token is no longer valid.
    invalid_reason: str

    #: The date and time the token was last updated, in ISO-8601 format.
    last_updated: str

    #: The date and time the token was last used, in ISO-8601 format.
    #:
    #: If the token has never been used, this will be ``None``.
    last_used: str | None

    #: A user-provided note about the token.
    note: str

    #: The access policies for the token.
    policy: JSONDict

    #: The date and time the token was created, in ISO-8601 format.
    time_added: str

    #: The token value.
    token: str

    #: The ID of the token generator that generated the token.
    token_generator_id: str

    #: Whether the token is currently valid.
    valid: bool


@resource_mimetype('application/vnd.reviewboard.org.api-tokens')
class APITokenListResource(ListResource[APITokenItemResource]):
    """List resource for API tokens.

    This corresponds to Review Board's
    :ref:`rb:webapi2.0-api-token-list-resource`.

    Version Added:
        6.0
    """
