"""Sphinx plugins for special links in the Release Notes."""

from __future__ import annotations

from docutils import nodes, utils


def setup(app):
    app.add_config_value('bugtracker_url', '', True)
    app.add_role('bug', bug_role)
    app.add_crossref_type(directivename='rbtcommand',
                          rolename='rbtcommand',
                          indextemplate='pair: %s; RBTools command')
    app.add_crossref_type(
        directivename=str('rbtconfig'),
        rolename=str('rbtconfig'),
        indextemplate=str('pair: %s; .reviewboardrc setting'))


def bug_role(role, rawtext, text, linenum, inliner, options={}, content=[]):
    try:
        bugnum = int(text)
        if bugnum <= 0:
            raise ValueError
    except ValueError:
        msg = inliner.reporter.error(
            'Bug number must be a number greater than or equal to 1; '
            '"%s" is invalid.' % text,
            line=linenum)
        prb = inliner.problematic(rawtext, rawtext, msg)
        return [prb], [msg]

    bugtracker_url = inliner.document.settings.env.config.bugtracker_url

    if not bugtracker_url or '%s' not in bugtracker_url:
        msg = inliner.reporter.error('bugtracker_url must be configured.',
                                     line=linenum)
        prb = inliner.problematic(rawtext, rawtext, msg)
        return [prb], [msg]

    ref = bugtracker_url % bugnum
    node = nodes.reference(rawtext, 'Bug #' + utils.unescape(text),
                           refuri=ref, **options)

    return [node], []
